const { app, BrowserWindow, shell } = require("electron");
const path = require("path");
const { createServer } = require("./server");

let mainWindow = null;
let httpServer = null;
let baseUrl = null;

function startServer() {
  const userDataDir = app.getPath("userData");
  const staticDir = path.join(__dirname, "app");
  const seedDbPath = path.join(__dirname, "assets", "demo_db.json");
  const { server } = createServer({ staticDir, seedDbPath, userDataDir });
  httpServer = server;

  return new Promise((resolve, reject) => {
    server.once("error", reject);
    server.listen(0, "127.0.0.1", () => {
      const address = server.address();
      const port = typeof address === "object" && address ? address.port : 0;
      resolve({ port });
    });
  });
}

function createWindow(urlBase) {
  mainWindow = new BrowserWindow({
    width: 1280,
    height: 820,
    minWidth: 1080,
    minHeight: 720,
    show: false,
    autoHideMenuBar: true,
    backgroundColor: "#f7f9fc",
    webPreferences: {
      contextIsolation: true,
      nodeIntegration: false,
      sandbox: true,
    },
  });

  mainWindow.loadURL(`${urlBase}/login.html?demo=1&autologin=1`);

  mainWindow.once("ready-to-show", () => {
    mainWindow.show();
  });

  const handleExternal = (targetUrl) => {
    if (!targetUrl.startsWith(urlBase)) {
      shell.openExternal(targetUrl);
      return true;
    }
    return false;
  };

  mainWindow.webContents.setWindowOpenHandler(({ url }) => {
    return handleExternal(url) ? { action: "deny" } : { action: "allow" };
  });

  mainWindow.webContents.on("will-navigate", (event, targetUrl) => {
    if (handleExternal(targetUrl)) {
      event.preventDefault();
    }
  });

  mainWindow.on("closed", () => {
    mainWindow = null;
  });
}

app.whenReady().then(async () => {
  try {
    const { port } = await startServer();
    baseUrl = `http://127.0.0.1:${port}`;
    createWindow(baseUrl);
  } catch (err) {
    console.error("Failed to start demo server:", err);
    app.quit();
  }
});

app.on("activate", () => {
  if (mainWindow === null && baseUrl) {
    createWindow(baseUrl);
  }
});

app.on("before-quit", () => {
  if (httpServer) {
    httpServer.close();
    httpServer = null;
  }
});

app.on("window-all-closed", () => {
  if (process.platform !== "darwin") {
    app.quit();
  }
});
