﻿import tkinter as tk
from tkinter import ttk, messagebox
from tkcalendar import DateEntry, Calendar
from demo_store import DemoError, DEMO_HORIZON_DAYS, login as demo_login, request as demo_request
import json
from datetime import datetime, timedelta, date
import threading
import math
from statistics import mean
import re  # for PB time validation
import calendar

token = None
role = None
current_user = None

# --- Dark UI palette ---
BG = "#131314"
SIDEBAR_BG = "#1b1b1b"
TOP_BG = "#1d1f22"
CARD_BG = "#1f1f21"
ACCENT = "#3A8CFF"
TEXT = "#ffffff"
TEXT_SECOND = "#a8a8a8"
TRAINING_COLORS = {
    "plávanie": "#3aaaff",
    "silový": "#9e4aff",
    "objemový": "#47ff9e",
    "technika": "#ffd84a",
    "oddych": "#8a8a8a",
}
TRAINING_TYPE_COLORS = {
    "aerobic": "#2ecc71",
    "threshold": "#f39c12",
    "speed": "#e74c3c",
    "technique": "#3498db",
    "strength": "#9b59b6",
    "recovery": "#95a5a6",
    "race": "#000000",
}

# --- Personal Best (PB) time validation ---
# Accepts:
#  - mm:ss.xx    (e.g., 00:27.55, 02:12.34)
#  - h:mm:ss.xx  (e.g., 1:52:03.47)
PB_TIME_RE = re.compile(r'^((\\d+):)?([0-5]?\\d):([0-5]\\d)[\\.,](\\d{2})$')

def valid_pb_time(text: str) -> bool:
    """Return True if text matches mm:ss.xx or h:mm:ss.xx, with minutes/seconds 0–59."""
    if not text:
        return True  # empty allowed; we validate only non-empty values
    m = PB_TIME_RE.match(text.strip())
    return m is not None

def normalize_pb_time(text: str) -> str:
    """
    Normalize to mm:ss.xx or h:mm:ss.xx with leading zeros.
    - If hours present -> h:mm:ss.xx (h no leading zero)
    - If no hours -> mm:ss.xx (mm two digits)
    """
    if not text:
        return text
    m = PB_TIME_RE.match(text.strip())
    if not m:
        return text
    _full, hours_grp, minutes_grp, seconds_grp, hundredths = m.group(0, 2, 3, 4, 5)
    if hours_grp is not None:
        h = int(hours_grp)
        mm = int(minutes_grp)
        ss = int(seconds_grp)
        cc = hundredths
        return f"{h}:{mm:02d}:{ss:02d}.{cc}"
    else:
        mm = int(minutes_grp)
        ss = int(seconds_grp)
        cc = hundredths
        return f"{mm:02d}:{ss:02d}.{cc}"


# =====================================================================
# 1. KONVERZIA ČASU 00:28.50 -> SEKUNDY
# =====================================================================
def time_to_seconds(time_str):
    # formát: mm:ss.xx alebo ss.xx
    try:
        parts = time_str.split(":")
        if len(parts) == 2:
            minutes = int(parts[0])
            seconds = float(parts[1])
            return minutes * 60 + seconds
        else:
            return float(time_str)
    except Exception:
        return None


# =====================================================================
# 2. AUTOMATICKÝ VÝPOČET OBJEMU Z POPISU TRÉNINGU
# =====================================================================
def extract_volume(description):
    """
    Nájde všetky vzory typu:
    - 400
    - 10x50
    - 5x 75
    - 3 × 100
    - 200R, 400R
    """

    total = 0

    # 1. vzory typu "10x50" alebo "5 x 75"
    pattern_sets = re.findall(r'(\d+)\s*[xX×]\s*(\d+)', description)
    for count, dist in pattern_sets:
        total += int(count) * int(dist)

    # 2. samostatné metre (400R, 200, 100)
    pattern_single = re.findall(r'(\d{2,4})\s*[Rr]?', description)
    for meters in pattern_single:
        total += int(meters)

    return total


# =====================================================================
# 3. AUTOMATICKÝ VÝPOČET KVALITY TRÉNINGU
# =====================================================================
def extract_quality(description):
    text = description.lower()

    if "95" in text or "naplno" in text or "svs" in text:
        return 0.95
    if "90" in text or "stup" in text:
        return 0.85
    if "tech" in text or "reg" in text:
        return 0.65

    return 0.75


# =====================================================================
# 4. PB IMPROVEMENT ALGORITMUS
# =====================================================================
def pb_improvement_score(trainings, days_to_competition, pb_seconds):
    """Legacy helper: returns only the final PB improvement percentage."""
    score, _ = pb_improvement_details(trainings, days_to_competition, pb_seconds)
    return score


def pb_improvement_details(trainings, days_to_competition, pb_seconds):
    """
    Return (score_percent, breakdown_dict) for the PB improvement algorithm.

    Trainings = list(dict):
        {
            "quality": 0-1,
            "delta_pb": seconds above PB,
            "fatigue": 0-1 (zatím fixne 0.75)
        }
    """

    T = len(trainings)
    if T < 3:
        return 0.0, {
            "form": 0.0,
            "trend": 0.0,
            "recovery": 0.0,
            "dopt": math.exp(-0.1 * abs(days_to_competition - 7)),
            "weighted_score": 0.0
        }

    # ---- 1. FORM ----
    weighted_sum = 0
    weight_total = 0
    for i, t in enumerate(trainings, start=1):
        w = i / T
        weighted_sum += t["quality"] * w
        weight_total += w
    Form = weighted_sum / weight_total if weight_total else 0

    # ---- 2. TREND ----
    first3 = [t["delta_pb"] for t in trainings[:3]]
    last3 = [t["delta_pb"] for t in trainings[-3:]]

    mean_first = mean(first3)
    mean_last = mean(last3)

    if mean_first == 0:
        Itrend = 1
    else:
        Itrend = 1 - (mean_last / mean_first)

    Itrend = max(0, min(1, Itrend))

    # ---- 3. RECOVERY (zatím fixed) ----
    Recovery = mean([t["fatigue"] for t in trainings])

    # ---- 4. OPTIMAL DAYS ----
    Dopt = math.exp(-0.1 * abs(days_to_competition - 7))

    # ---- 5. FINAL ----
    PB_Improvement = (
        0.35 * Form +
        0.30 * Itrend +
        0.20 * Recovery +
        0.15 * Dopt
    )

    return PB_Improvement * 100, {
        "form": Form,
        "trend": Itrend,
        "recovery": Recovery,
        "dopt": Dopt,
        "weighted_score": PB_Improvement * 100
    }


# -------------------- HTTP API --------------------
def api_request(method, endpoint, data=None):
    """Handles communication with the local demo store."""
    if method not in {"GET", "POST", "PUT", "DELETE"}:
        raise ValueError("Invalid method")
    return demo_request(method, endpoint, data, current_user=current_user, role=role)


def api_login(username, password):
    """Validate local demo credentials and store role."""
    global token, role, current_user
    user = demo_login(username, password)
    token = f"demo-{user.username}"
    role = user.role  # "coach" or "trainee"
    current_user = user.username


# -------------------- APP --------------------
class TrainingApp:
    def __init__(self, root):
        self.root = root
        self.root.title("Online Training Calendar (Demo Version)")
        self.root.geometry("1180x740")
        self.root.configure(bg="#f0f2f5")
        self.root.tk.call('tk', 'scaling', 1.2)

        # State
        self.all_trainings = []           # last fetch from server
        self.training_cache = {}          # id -> training dict
        self.selected_training_id = None  # selected item (from popup)
        self.popup = None
        self.popup_listbox = None         # popup list widget
        self.popup_id_by_index = {}       # popup index -> training id
        self.pb_window = None             # PB predictor window
        self.pb_username_var = None
        self.pb_pb_var = None
        self.pb_days_var = None
        self.pb_result_var = None
        self.pb_detail_btn = None
        self.pb_last_score = None
        self.pb_last_breakdown = None
        self.dark_win = None
        self.dark_main_area = None
        self.dark_month_label = None
        self.dark_grid = None
        self.dark_month = datetime.now().replace(day=1)
        self.api_dark_win = None
        self.api_dark_area = None
        self.api_dark_year = datetime.now().year
        self.api_dark_month = datetime.now().month
        self.api_month_trainings = {}

        # Coach state
        self.trainees = []
        self.selected_trainee = tk.StringVar(value="")
        self.selected_calendar_date: str | None = None

        # Activities / disciplines
        self.activities = [
            "Gym Training", "Cardio", "Running", "Cycling", "Swimming", "HIIT", "Yoga", "Pilates",
            "CrossFit", "Stretching", "Weightlifting", "Powerlifting", "Football", "Basketball",
            "Volleyball", "Tennis", "Badminton", "Table Tennis", "Hockey", "Rugby", "Cricket",
            "Hiking", "Rock Climbing", "Rowing", "Kayaking", "Skiing", "Snowboarding",
            "Skateboarding", "Boxing", "Kickboxing", "Martial Arts", "Judo", "Karate",
            "Taekwondo", "Wrestling", "Meditation", "Breathing Exercises", "Mobility Training",
            "Recovery Session"
        ]
        self.training_types = ["aerobic", "anaerobic", "technique", "recovery"]
        # Common swimming disciplines (editable in UI for the profile)
        self.disciplines = [
            "50 Free", "100 Free", "200 Free", "400 Free", "800 Free", "1500 Free",
            "50 Back", "100 Back", "200 Back",
            "50 Breast", "100 Breast", "200 Breast",
            "50 Fly", "100 Fly", "200 Fly",
            "200 IM", "400 IM",
            "Open Water", "Other"
        ]
        # Order used for Personal Bests
        self.default_pbs_order = [
            "50 Free", "100 Free", "200 Free", "400 Free", "800 Free", "1500 Free",
            "50 Back", "100 Back", "200 Back",
            "50 Breast", "100 Breast", "200 Breast",
            "50 Fly", "100 Fly", "200 Fly",
            "200 IM", "400 IM"
        ]

        self.selected_activity = tk.StringVar(value=self.activities[0])

        self.setup_styles()
        self.create_login_frame()
        self.create_main_frame()

        self.main_frame.pack_forget()
        self.login_frame.pack(expand=True, fill="both")
        self.auto_refresh_threaded()

    # ---------- Thread-safe UI helpers ----------
    def ui(self, fn, *args, **kwargs):
        """Run any Tk operation on the main thread."""
        self.root.after(0, lambda: fn(*args, **kwargs))

    def ui_msg(self, kind, title, text):
        """Show messageboxes safely from any thread."""
        if kind == "info":
            self.ui(messagebox.showinfo, title, text)
        elif kind == "warn":
            self.ui(messagebox.showwarning, title, text)
        else:
            self.ui(messagebox.showerror, title, text)

    # ---------- Styles ----------
    def setup_styles(self):
        style = ttk.Style()
        style.theme_use("clam")
        style.configure("TButton", padding=8, relief="flat")
        style.configure("Green.TButton", background="#2ecc71", foreground="black")
        style.configure("Red.TButton", background="#e74c3c", foreground="black")
        style.configure("Blue.TButton", background="#3498db", foreground="black")
        style.configure("Purple.TButton", background="#9b59b6", foreground="black")
        # NEW: highlight invalid time inputs softly
        style.configure("Invalid.TEntry", fieldbackground="#ffe6e6")
        # Inputs: slightly taller, clearer font
        style.configure("Big.TEntry", padding=6, font=("Arial", 12))
        style.configure("Big.TCombobox", padding=6, font=("Arial", 12))

    def set_crud_buttons_state(self, state: str):
        """Enable/disable Add/Edit/Delete. For coaches, Edit/Delete stay enabled even without selection."""
        if getattr(self, "add_btn", None):
            self.add_btn.configure(state=state)
        coach_enabled = (state == "normal") and (role == "coach")
        if getattr(self, "edit_btn", None):
            self.edit_btn.configure(state=("normal" if coach_enabled else "disabled"))
        if getattr(self, "delete_btn", None):
            self.delete_btn.configure(state=("normal" if coach_enabled else "disabled"))
        if getattr(self, "generate_training_btn", None):
            self.generate_training_btn.configure(state=("normal" if coach_enabled else "disabled"))

    # ---------- Login ----------
    def create_login_frame(self):
        self.login_frame = tk.Frame(self.root, bg="#f0f2f5")
        card = tk.Frame(self.login_frame, bg="white", highlightbackground="#dcdcdc", highlightthickness=1)
        card.place(relx=0.5, rely=0.5, anchor="center", width=400, height=420)

        tk.Label(card, text="Welcome", bg="white", font=("Arial", 24, "bold")).pack(pady=20)
        tk.Label(card, text="Training Calendar — DEMO VERSION", bg="white", font=("Arial", 18, "bold")).pack(pady=(0, 20))

        self.username_entry = ttk.Entry(card, foreground="gray")
        self.username_entry.pack(pady=10, padx=40, fill="x")
        self.username_entry.insert(0, "Username")
        self.username_entry.bind("<FocusIn>", lambda e: self._clear_placeholder(self.username_entry, "Username"))
        self.username_entry.bind("<FocusOut>", lambda e: self._add_placeholder(self.username_entry, "Username"))

        self.password_entry = ttk.Entry(card, show="", foreground="gray")
        self.password_entry.pack(pady=10, padx=40, fill="x")
        self.password_entry.insert(0, "Password")
        self.password_entry.bind("<FocusIn>", self._toggle_password)
        self.password_entry.bind("<FocusOut>", self._restore_password_placeholder)

        ttk.Button(card, text="Login", style="Green.TButton", command=self.threaded_login).pack(pady=20)

    def _clear_placeholder(self, entry, text):
        if entry.get() == text:
            entry.delete(0, "end")
            entry.config(foreground="black")

    def _add_placeholder(self, entry, text):
        if not entry.get():
            entry.insert(0, text)
            entry.config(foreground="gray")

    def _toggle_password(self, _):
        if self.password_entry.get() == "Password":
            self.password_entry.delete(0, "end")
            self.password_entry.config(show="*", foreground="black")

    def _restore_password_placeholder(self, _):
        if not self.password_entry.get():
            self.password_entry.insert(0, "Password")
            self.password_entry.config(show="", foreground="gray")

    def threaded_login(self):
        threading.Thread(target=self.login_action, daemon=True).start()

    def login_action(self):
        username = self.username_entry.get().strip()
        password = self.password_entry.get().strip()
        if not username or username == "Username" or not password or password == "Password":
            self.ui_msg("warn", "Login", "Please enter valid credentials.")
            return
        try:
            api_login(username, password)
            def after_ok():
                self.login_frame.pack_forget()
                self.main_frame.pack(expand=True, fill="both")
                self.set_crud_buttons_state("normal" if role == "coach" else "disabled")
                self.profile_btn.config(text="Edit Profile (Coach)" if role == "coach" else "View Profile")
                if role == "coach":
                    if not self.add_trainee_btn.winfo_manager():
                        self.add_trainee_btn.pack(fill="x", pady=4)
                    self.load_trainees_threaded()
                else:
                    self.add_trainee_btn.pack_forget()
                self.refresh_list_threaded()
            self.ui(after_ok)
        except DemoError as e:
            self.ui_msg("error", "Login Failed", str(e))
        except Exception as e:
            self.ui_msg("error", "Login Failed", str(e))

    # ---------- Main Layout ----------
    def create_main_frame(self):
        self.main_frame = tk.Frame(self.root, bg="#f0f2f5")
        self.main_frame.pack(fill="both", expand=True)

        # LEFT: sidebar
        self.left_frame = tk.Frame(self.main_frame, bg="white", width=360, highlightbackground="#dcdcdc", highlightthickness=1)
        self.left_frame.pack(side="left", fill="y", padx=(10, 5), pady=10)
        self.left_frame.pack_propagate(False)

        # RIGHT: calendar
        self.right_frame = tk.Frame(self.main_frame, bg="white", highlightbackground="#dcdcdc", highlightthickness=1)
        self.right_frame.pack(side="left", fill="both", expand=True, padx=(5, 10), pady=10)

        # Coach: Trainee selector
        self.coach_top = tk.Frame(self.left_frame, bg="white")
        self.coach_top.pack(fill="x", padx=12, pady=(10, 0))
        self.coach_top.pack_forget()  # shown after login if coach

        tk.Label(self.coach_top, text="Trainee:", bg="white").pack(anchor="w")
        self.user_select = ttk.Combobox(self.coach_top, textvariable=self.selected_trainee, state="readonly")
        self.user_select.pack(fill="x", pady=5)
        self.user_select.bind("<<ComboboxSelected>>", lambda e: self.on_trainee_changed())

        # Buttons
        tk.Label(self.left_frame, text="Actions", bg="white", font=("Arial", 12, "bold")).pack(pady=(10, 4))
        btns_frame = tk.Frame(self.left_frame, bg="white")
        btns_frame.pack(padx=12, pady=8, fill="x")

        self.add_btn = ttk.Button(btns_frame, text="Add Training", style="Green.TButton",
                                  command=self.add_training_threaded)
        self.edit_btn = ttk.Button(btns_frame, text="Edit Training", style="Purple.TButton", command=self.open_edit_window)
        self.delete_btn = ttk.Button(btns_frame, text="Delete Training", style="Red.TButton",
                                     command=self.delete_training_threaded)
        self.refresh_btn = ttk.Button(btns_frame, text="Refresh", style="Blue.TButton",
                                      command=self.refresh_list_threaded)
        self.pb_predict_btn = ttk.Button(btns_frame, text="PB Improvement", command=self.open_pb_predictor_window)
        self.dark_ui_btn = ttk.Button(btns_frame, text="Dark UI", style="Blue.TButton", command=self.open_dark_ui)
        self.api_dark_ui_btn = ttk.Button(btns_frame, text="API Dark UI", style="Blue.TButton", command=self.open_api_dark_ui)
        self.generate_training_btn = ttk.Button(btns_frame, text="Generate with AI", style="Blue.TButton", command=self.open_generate_training_window)
        self.profile_btn = ttk.Button(btns_frame, text="Edit Profile (Coach)", command=self.open_profile_window)
        self.weekly_graph_btn = ttk.Button(btns_frame, text="Týždenný graf", command=self.show_weekly_graph)
        for w in (self.add_btn, self.edit_btn, self.delete_btn, self.refresh_btn, self.pb_predict_btn, self.dark_ui_btn, self.api_dark_ui_btn, self.generate_training_btn, self.weekly_graph_btn, self.profile_btn):
            w.pack(fill="x", pady=4)

        self.add_trainee_btn = ttk.Button(btns_frame, text="Add Trainee", command=self.open_add_trainee_window)
        self.add_trainee_btn.configure(state="disabled")

        ttk.Separator(self.left_frame, orient="horizontal").pack(fill="x", padx=12, pady=10)
        ttk.Button(self.left_frame, text="Logout", style="Red.TButton", command=self.logout_action).pack(fill="x", padx=12, pady=(0, 12))

        # Add form
        input_frame = tk.Frame(self.left_frame, bg="white")
        input_frame.pack(padx=12, pady=(10, 20), fill="x")

        tk.Label(input_frame, text="Date:", bg="white").pack(anchor="w")
        self.cal = DateEntry(input_frame, date_pattern="yyyy-mm-dd", font=("Arial", 12))
        self.cal.pack(fill="x", pady=5, ipady=4)

        tk.Label(input_frame, text="Time (HH:MM):", bg="white").pack(anchor="w")
        self.time_entry = ttk.Entry(input_frame, style="Big.TEntry")
        self.time_entry.pack(fill="x", pady=5)

        tk.Label(input_frame, text="Sport:", bg="white").pack(anchor="w")
        self.activity_combo = ttk.Combobox(
            input_frame,
            textvariable=self.selected_activity,
            values=self.activities,
            state="readonly",
            style="Big.TCombobox",
        )
        self.activity_combo.pack(fill="x", pady=5)

        tk.Label(input_frame, text="Description:", bg="white").pack(anchor="w")
        desc_wrap = tk.Frame(input_frame, bg="white", highlightbackground="#dcdcdc", highlightthickness=1)
        desc_wrap.pack(fill="both", pady=5, expand=True)
        desc_wrap.configure(height=180)
        desc_wrap.pack_propagate(False)  # keep space visible
        self.desc_text = tk.Text(
            desc_wrap,
            height=10,
            wrap="word",
            bg="#000000",
            fg="#ffffff",
            insertbackground="#ffffff",
        )
        desc_scroll = ttk.Scrollbar(desc_wrap, orient="vertical", command=self.desc_text.yview)
        self.desc_text.configure(yscrollcommand=desc_scroll.set)
        self.desc_text.pack(side="left", fill="both", expand=True)
        desc_scroll.pack(side="right", fill="y")

        # Calendar
        tk.Label(self.right_frame, text="Calendar (DEMO VERSION)", bg="white", font=("Arial", 13, "bold")).pack(pady=(10, 0))
        self.main_cal = Calendar(
            self.right_frame,
            selectmode="day",
            date_pattern="yyyy-mm-dd",
            showweeknumbers=False,
            background="white",
            foreground="black",
            tooltipbackground="#ffffe0",
        )
        self.main_cal.pack(fill="both", expand=True, padx=20, pady=20)
        self.main_cal.bind("<<CalendarSelected>>", self.on_main_calendar_select)
        # initial selected date
        try:
            self._set_selected_calendar_date(self.main_cal.get_date())
        except Exception:
            self.selected_calendar_date = None

    # ---------- Dark UI (preview) ----------
    def open_dark_ui(self):
        """Show a dark-mode dashboard with a custom month grid."""
        if self.dark_win and self.dark_win.winfo_exists():
            self.dark_win.lift()
            return

        self.dark_month = datetime.now().replace(day=1)
        win = tk.Toplevel(self.root)
        self.dark_win = win
        win.title("Training Analytics — Dark UI")
        win.geometry("1280x800")
        win.configure(bg=BG)
        win.protocol("WM_DELETE_WINDOW", self._close_dark_ui)

        container = tk.Frame(win, bg=BG)
        container.pack(fill="both", expand=True)

        sidebar = tk.Frame(container, bg=SIDEBAR_BG, width=220)
        sidebar.pack(side="left", fill="y")
        sidebar.pack_propagate(False)

        tk.Label(
            sidebar,
            text="MENU",
            fg=TEXT_SECOND,
            bg=SIDEBAR_BG,
            font=("Segoe UI", 14, "bold"),
            pady=20,
        ).pack()

        buttons = [
            ("Dashboard", lambda: self.refresh_dark_view()),
            ("Kalendár", lambda: self._render_dark_month(self.dark_month.year, self.dark_month.month)),
            ("Zverenci", self.load_trainees_threaded),
            ("Analytika", self.show_weekly_graph),
            ("Importy", self.open_profile_window),
            ("Nastavenia", lambda: messagebox.showinfo("Nastavenia", "Pripravujeme.")),
        ]

        for text, command in buttons:
            btn = tk.Button(
                sidebar,
                text=text,
                command=command,
                font=("Segoe UI", 13),
                fg=TEXT,
                bg=SIDEBAR_BG,
                activebackground=ACCENT,
                activeforeground=TEXT,
                relief="flat",
                bd=0,
                pady=8,
            )
            btn.pack(fill="x", padx=20, pady=3)

        main_col = tk.Frame(container, bg=BG)
        main_col.pack(side="left", fill="both", expand=True)

        topbar = tk.Frame(main_col, bg=TOP_BG, height=60)
        topbar.pack(side="top", fill="x")

        tk.Button(
            topbar,
            text="‹",
            font=("Segoe UI", 14, "bold"),
            fg=TEXT,
            bg=TOP_BG,
            activebackground=ACCENT,
            activeforeground=TEXT,
            relief="flat",
            bd=0,
            command=lambda: self._change_dark_month(-1),
        ).pack(side="left", padx=(12, 4), pady=10)

        tk.Button(
            topbar,
            text="›",
            font=("Segoe UI", 14, "bold"),
            fg=TEXT,
            bg=TOP_BG,
            activebackground=ACCENT,
            activeforeground=TEXT,
            relief="flat",
            bd=0,
            command=lambda: self._change_dark_month(1),
        ).pack(side="left", padx=(0, 12), pady=10)

        app_title = tk.Label(
            topbar,
            text="📅 Tréningový kalendár",
            fg=TEXT,
            bg=TOP_BG,
            font=("Segoe UI", 16, "bold"),
        )
        app_title.pack(side="left", padx=10)

        tk.Button(
            topbar,
            text="Dnes",
            font=("Segoe UI", 12),
            fg=TEXT,
            bg=ACCENT,
            relief="flat",
            padx=20,
            command=self._reset_dark_month,
        ).pack(side="right", padx=10, pady=10)

        self.dark_main_area = tk.Frame(main_col, bg=BG)
        self.dark_main_area.pack(side="top", fill="both", expand=True)
        self._render_dark_month(self.dark_month.year, self.dark_month.month)

    def _reset_dark_month(self):
        self.dark_month = datetime.now().replace(day=1)
        self._render_dark_month(self.dark_month.year, self.dark_month.month)

    def _change_dark_month(self, delta):
        base = self.dark_month
        new_month_index = base.month - 1 + delta
        new_year = base.year + new_month_index // 12
        new_month = (new_month_index % 12) + 1
        self.dark_month = datetime(new_year, new_month, 1)
        self._render_dark_month(new_year, new_month)

    def _render_dark_month(self, year, month):
        if not self.dark_main_area or not self.dark_main_area.winfo_exists():
            return

        for widget in self.dark_main_area.winfo_children():
            widget.destroy()

        header = tk.Label(
            self.dark_main_area,
            text=f"{calendar.month_name[month]} {year}",
            fg=TEXT,
            bg=BG,
            font=("Segoe UI", 22, "bold"),
            pady=20,
        )
        header.pack()
        self.dark_month_label = header

        grid = tk.Frame(self.dark_main_area, bg=BG)
        grid.pack()
        self.dark_grid = grid

        cal = calendar.Calendar()
        for r, week in enumerate(cal.monthdatescalendar(year, month)):
            for c, day in enumerate(week):
                in_month = day.month == month
                day_key = day.strftime("%Y-%m-%d")
                trainings = [t for t in self.all_trainings if t.get("date") == day_key]
                self._create_dark_day_card(grid, day, r, c, in_month, trainings)

    def _create_dark_day_card(self, parent, day, row, col, in_month, trainings):
        base_bg = CARD_BG if in_month else "#1a1a1a"
        border = ACCENT if trainings else "#2b2b2b"
        card = tk.Frame(
            parent,
            bg=base_bg,
            width=140,
            height=110,
            highlightbackground=border,
            highlightthickness=1,
        )
        card.grid(row=row, column=col, padx=6, pady=6)
        card.grid_propagate(False)

        day_label = tk.Label(
            card,
            text=str(day.day),
            fg=TEXT if in_month else TEXT_SECOND,
            bg=base_bg,
            font=("Segoe UI", 15, "bold"),
        )
        day_label.pack(anchor="nw", padx=8, pady=5)

        if trainings:
            main_tr = trainings[0]
            sport = main_tr.get("sport") or "Tréning"
            desc = (main_tr.get("description", "") or "").strip().replace("\n", " ")
            if len(desc) > 34:
                desc = desc[:31] + "..."
            tk.Label(
                card,
                text=f"• {sport}",
                fg=ACCENT,
                bg=base_bg,
                font=("Segoe UI", 11, "bold"),
            ).pack(anchor="w", padx=8, pady=(0, 2))
            tk.Label(
                card,
                text=desc or "Naplánovaný tréning",
                fg=TEXT_SECOND,
                bg=base_bg,
                wraplength=120,
                justify="left",
                font=("Segoe UI", 10),
            ).pack(anchor="w", padx=8)
            if len(trainings) > 1:
                tk.Label(
                    card,
                    text=f"+{len(trainings) - 1} ďalšie",
                    fg=TEXT_SECOND,
                    bg=base_bg,
                    font=("Segoe UI", 9),
                ).pack(anchor="w", padx=8, pady=(3, 0))
        else:
            tk.Label(
                card,
                text="— voľno —",
                fg=TEXT_SECOND,
                bg=base_bg,
                font=("Segoe UI", 10),
            ).pack(expand=True)

        for widget in (card, day_label, *card.winfo_children()):
            widget.bind("<Button-1>", lambda _e, d=day: self._on_dark_day_click(d))

    def _on_dark_day_click(self, day):
        """Sync date selection with the form and open the popup list."""
        try:
            self.cal.set_date(day)
        except Exception:
            pass
        self.open_day_popup(day)

    def refresh_dark_view(self):
        if self.dark_win and self.dark_win.winfo_exists():
            self._render_dark_month(self.dark_month.year, self.dark_month.month)

    def _close_dark_ui(self):
        if self.dark_win and self.dark_win.winfo_exists():
            self.dark_win.destroy()
        self.dark_win = None
        self.dark_main_area = None
        self.dark_month_label = None
        self.dark_grid = None

    # ---------- API-connected Dark UI ----------
    def open_api_dark_ui(self):
        """Full-screen dark UI that pulls calendar/dashboard data from the API."""
        if not token:
            messagebox.showwarning("Login required", "Please log in first.")
            return
        if self.api_dark_win and self.api_dark_win.winfo_exists():
            self.api_dark_win.lift()
            self._api_show_calendar()
            return

        self.api_dark_year = datetime.now().year
        self.api_dark_month = datetime.now().month
        self.api_month_trainings = {}

        win = tk.Toplevel(self.root)
        self.api_dark_win = win
        win.title("Training Analytics — API Connected")
        win.geometry("1400x850")
        win.configure(bg=BG)
        win.protocol("WM_DELETE_WINDOW", self._close_api_dark_ui)

        sidebar = tk.Frame(win, bg=SIDEBAR_BG, width=220)
        sidebar.pack(side="left", fill="y")
        sidebar.pack_propagate(False)
        tk.Label(
            sidebar,
            text="MENU",
            fg=TEXT_SECOND,
            bg=SIDEBAR_BG,
            font=("Segoe UI", 14, "bold"),
            pady=20,
        ).pack()

        buttons = [
            ("Dashboard", self._api_show_dashboard),
            ("Kalendár", self._api_show_calendar),
            ("Zverenci", lambda: self.temp_api_panel("Zverenci")),
            ("Analytika", lambda: self.temp_api_panel("Analytika")),
            ("Importy", lambda: self.temp_api_panel("Importy")),
            ("Nastavenia", self._api_show_settings),
        ]
        for text, command in buttons:
            tk.Button(
                sidebar,
                text=text,
                command=command,
                font=("Segoe UI", 13),
                fg=TEXT,
                bg=SIDEBAR_BG,
                activebackground=ACCENT,
                activeforeground=TEXT,
                relief="flat",
                bd=0,
                pady=8,
            ).pack(fill="x", padx=20, pady=3)

        topbar = tk.Frame(win, bg=TOP_BG, height=60)
        topbar.pack(side="top", fill="x")

        tk.Label(
            topbar,
            text="Training Analytics (API)",
            fg=TEXT,
            bg=TOP_BG,
            font=("Segoe UI", 18, "bold"),
        ).pack(side="left", padx=20)

        tk.Button(
            topbar,
            text="Dnes",
            fg=TEXT,
            bg=ACCENT,
            font=("Segoe UI", 12),
            relief="flat",
            padx=20,
            command=self._api_today,
        ).pack(side="right", padx=10, pady=10)

        self.api_dark_area = tk.Frame(win, bg=BG)
        self.api_dark_area.pack(side="right", fill="both", expand=True)
        self._api_show_calendar()

    def _close_api_dark_ui(self):
        if self.api_dark_win and self.api_dark_win.winfo_exists():
            self.api_dark_win.destroy()
        self.api_dark_win = None
        self.api_dark_area = None

    def _api_today(self):
        self.api_dark_year = datetime.now().year
        self.api_dark_month = datetime.now().month
        self._api_show_calendar()

    def _api_clear(self):
        if not self.api_dark_area or not self.api_dark_area.winfo_exists():
            return
        for w in self.api_dark_area.winfo_children():
            w.destroy()

    def _api_get(self, endpoint):
        try:
            return demo_request(
                "GET",
                endpoint,
                current_user=current_user,
                role=role,
                target_username=self.get_target_username(),
            )
        except DemoError as e:
            messagebox.showerror("Demo Error", str(e))
            return None
        except Exception as e:
            messagebox.showerror("Demo Error", str(e))
            return None

    def _api_post(self, endpoint, payload):
        try:
            return demo_request(
                "POST",
                endpoint,
                payload,
                current_user=current_user,
                role=role,
                target_username=self.get_target_username(),
            )
        except DemoError as e:
            messagebox.showerror("Demo Error", str(e))
            return None
        except Exception as e:
            messagebox.showerror("Demo Error", str(e))
            return None

    def _api_refresh_month(self):
        data = self._api_get(f"/calendar/{self.api_dark_year}/{self.api_dark_month}")
        if data is not None:
            self.api_month_trainings = data

    def _api_show_calendar(self):
        self._api_clear()
        self._api_refresh_month()

        tk.Label(
            self.api_dark_area,
            text=f"{calendar.month_name[self.api_dark_month]} {self.api_dark_year}",
            fg=TEXT,
            bg=BG,
            font=("Segoe UI", 24, "bold"),
        ).pack(pady=20)

        grid = tk.Frame(self.api_dark_area, bg=BG)
        grid.pack()

        cal = calendar.Calendar()
        for r, week in enumerate(cal.monthdatescalendar(self.api_dark_year, self.api_dark_month)):
            for c, day in enumerate(week):
                self._api_draw_day(grid, day, r, c)

    def _api_draw_day(self, parent, day, row, col):
        bg = CARD_BG if day.month == self.api_dark_month else "#1a1a1a"
        card = tk.Frame(
            parent,
            bg=bg,
            width=160,
            height=120,
            highlightbackground="#2b2b2b",
            highlightthickness=1,
        )
        card.grid(row=row, column=col, padx=8, pady=8)
        card.grid_propagate(False)

        def on_enter(_): card.config(bg="#26292b")
        def on_leave(_): card.config(bg=bg)
        card.bind("<Enter>", on_enter)
        card.bind("<Leave>", on_leave)

        tk.Label(
            card,
            text=day.day,
            fg=TEXT,
            bg=card["bg"],
            font=("Segoe UI", 15, "bold"),
        ).pack(anchor="nw", padx=8, pady=5)

        iso = day.isoformat()
        trainings = self.api_month_trainings.get(iso, [])
        for tr in trainings:
            color = TRAINING_COLORS.get(tr.get("type"), ACCENT)
            tk.Label(
                card,
                text=f"• {tr.get('name', 'Tréning')}",
                fg=color,
                bg=card["bg"],
                font=("Segoe UI", 11),
            ).pack(anchor="w", padx=10)

        card.bind("<Button-1>", lambda _e, d=day: self._api_open_day_detail(d))

    def _api_open_day_detail(self, day):
        iso = day.isoformat()
        data = self._api_get(f"/calendar/day/{iso}") or []

        win = tk.Toplevel(self.api_dark_win or self.root)
        win.title(f"Detail {iso}")
        win.geometry("450x450")
        win.configure(bg=BG)

        tk.Label(
            win,
            text=f"Tréningy {iso}",
            fg=TEXT,
            bg=BG,
            font=("Segoe UI", 16, "bold"),
        ).pack(pady=20)

        if not data:
            tk.Label(win, text="Žiadny tréning.", fg=TEXT_SECOND, bg=BG).pack()
            return

        for tr in data:
            frame = tk.Frame(win, bg=CARD_BG, pady=10, padx=10)
            frame.pack(fill="x", pady=10, padx=20)

            color = TRAINING_COLORS.get(tr.get("type"), ACCENT)
            tk.Label(
                frame,
                text=tr.get("name", "Tréning"),
                fg=color,
                bg=CARD_BG,
                font=("Segoe UI", 13, "bold"),
            ).pack(anchor="w")
            tk.Label(
                frame,
                text=f"Typ: {tr.get('type', 'nezadaný')}",
                fg=TEXT_SECOND,
                bg=CARD_BG,
            ).pack(anchor="w")

    def _api_show_dashboard(self):
        self._api_clear()
        data = self._api_get("/dashboard")
        if data is None:
            return

        tk.Label(
            self.api_dark_area,
            text="Dashboard",
            fg=TEXT,
            bg=BG,
            font=("Segoe UI", 28, "bold"),
        ).pack(pady=20)

        cards = tk.Frame(self.api_dark_area, bg=BG)
        cards.pack()

        stats = [
            ("Týždenná záťaž", data.get("weekly_load", "—")),
            ("Posledný PB", data.get("last_pb", "—")),
            ("Tréningy tento mesiac", data.get("monthly_sessions", "—")),
        ]

        for title, value in stats:
            frame = tk.Frame(cards, bg=CARD_BG, width=280, height=150)
            frame.pack(side="left", padx=20)
            frame.pack_propagate(False)
            tk.Label(
                frame,
                text=title,
                fg=TEXT_SECOND,
                bg=CARD_BG,
                font=("Segoe UI", 12),
            ).pack(anchor="w", padx=10, pady=10)
            tk.Label(
                frame,
                text=value,
                fg=ACCENT,
                bg=CARD_BG,
                font=("Segoe UI", 22, "bold"),
            ).pack(anchor="center")

        graph = tk.Frame(self.api_dark_area, bg=CARD_BG, width=900, height=350)
        graph.pack(pady=40)
        graph.pack_propagate(False)
        tk.Label(
            graph,
            text="Graf intenzity — dáta z API",
            fg=TEXT_SECOND,
            bg=CARD_BG,
            font=("Segoe UI", 14),
        ).pack(pady=20)
        zones = data.get("zone_distribution") or {}
        if zones:
            zone_line = " | ".join([f"{z}: {zones.get(z, 0)} m" for z in ["Z1", "Z2", "Z3", "Z4", "Z5"]])
            tk.Label(
                graph,
                text=zone_line,
                fg=TEXT,
                bg=CARD_BG,
                font=("Segoe UI", 12),
            ).pack(pady=(0, 12))

    def _api_show_settings(self):
        if role != "coach":
            messagebox.showwarning("Len pre trénera", "Nastavenia trénera sú dostupné iba pre trénerov.")
            return
        self._api_clear()
        settings = self._api_get("/api/coach/settings") or {}
        # Local fallback if API returns empty settings.
        defaults = {
            "training_style": "balanced",
            "be_cautious_on_absence": True,
            "no_volume_increase_on_absence": True,
            "gradual_return": True,
            "allow_hard_sessions": "often",
            "use_research": "advisory",
            "ai_explanation_level": "normal",
        }
        for key, value in defaults.items():
            settings.setdefault(key, value)

        card = tk.Frame(self.api_dark_area, bg=CARD_BG, padx=32, pady=24)
        card.pack(padx=40, pady=30, fill="both", expand=True)

        tk.Label(
            card,
            text="Nastavenia trénera",
            fg=TEXT,
            bg=CARD_BG,
            font=("Segoe UI", 22, "bold"),
        ).pack(anchor="w")
        tk.Label(
            card,
            text="Odporúčané nastaviť raz, ale môžeš meniť kedykoľvek. Platí od ďalšieho tréningu.",
            fg=TEXT_SECOND,
            bg=CARD_BG,
            font=("Segoe UI", 11),
            wraplength=760,
            justify="left",
        ).pack(anchor="w", pady=(6, 16))

        style_var = tk.StringVar(value=settings.get("training_style", "balanced"))
        cautious_var = tk.BooleanVar(value=bool(settings.get("be_cautious_on_absence", True)))
        no_volume_var = tk.BooleanVar(value=bool(settings.get("no_volume_increase_on_absence", True)))
        gradual_var = tk.BooleanVar(value=bool(settings.get("gradual_return", True)))
        hard_var = tk.StringVar(value=settings.get("allow_hard_sessions", "often"))
        research_var = tk.StringVar(value=settings.get("use_research", "advisory"))
        explain_var = tk.StringVar(value=settings.get("ai_explanation_level", "normal"))

        def section_title(text):
            tk.Label(
                card,
                text=text,
                fg=TEXT,
                bg=CARD_BG,
                font=("Segoe UI", 13, "bold"),
            ).pack(anchor="w", pady=(12, 4))

        section_title("Ako chceš trénovať?")
        for label, value in [("Skôr opatrne", "conservative"), ("Vyvážene", "balanced"), ("Skôr tvrdo", "aggressive")]:
            tk.Radiobutton(
                card,
                text=label,
                variable=style_var,
                value=value,
                fg=TEXT,
                bg=CARD_BG,
                selectcolor=CARD_BG,
                activebackground=CARD_BG,
                activeforeground=TEXT,
                font=("Segoe UI", 11),
            ).pack(anchor="w")
        tk.Label(
            card,
            text="Ovplyvňuje, ako rýchlo sa zvyšuje záťaž a ako opatrne sa postupuje po pauzách.",
            fg=TEXT_SECOND,
            bg=CARD_BG,
            font=("Segoe UI", 10),
            wraplength=760,
            justify="left",
        ).pack(anchor="w")

        section_title("Keď športovec ochorie alebo vynechá tréning")
        for label, var in [
            ("Netlačiť na výkon", cautious_var),
            ("Nezvyšovať objem", no_volume_var),
            ("Po návrate ísť postupne", gradual_var),
        ]:
            tk.Checkbutton(
                card,
                text=label,
                variable=var,
                fg=TEXT,
                bg=CARD_BG,
                selectcolor=CARD_BG,
                activebackground=CARD_BG,
                activeforeground=TEXT,
                font=("Segoe UI", 11),
            ).pack(anchor="w")
        tk.Label(
            card,
            text="Pomáha predchádzať zraneniam a preťaženiu.",
            fg=TEXT_SECOND,
            bg=CARD_BG,
            font=("Segoe UI", 10),
        ).pack(anchor="w")

        section_title("Ťažké tréningy")
        for label, value in [("Môžu byť bežne", "often"), ("Len výnimočne", "rare"), ("Radšej nie", "never")]:
            tk.Radiobutton(
                card,
                text=label,
                variable=hard_var,
                value=value,
                fg=TEXT,
                bg=CARD_BG,
                selectcolor=CARD_BG,
                activebackground=CARD_BG,
                activeforeground=TEXT,
                font=("Segoe UI", 11),
            ).pack(anchor="w")
        tk.Label(
            card,
            text="Ťažký tréning = vysoká záťaž alebo veľká únava.",
            fg=TEXT_SECOND,
            bg=CARD_BG,
            font=("Segoe UI", 10),
        ).pack(anchor="w")

        section_title("Odborné poznatky (výskum)")
        for label, value in [("Použiť ako odporúčanie", "advisory"), ("Nepoužívať", "off")]:
            tk.Radiobutton(
                card,
                text=label,
                variable=research_var,
                value=value,
                fg=TEXT,
                bg=CARD_BG,
                selectcolor=CARD_BG,
                activebackground=CARD_BG,
                activeforeground=TEXT,
                font=("Segoe UI", 11),
            ).pack(anchor="w")
        tk.Label(
            card,
            text="Ak je zapnuté, AI berie do úvahy overené poznatky, ale nikdy ich nevnucuje.",
            fg=TEXT_SECOND,
            bg=CARD_BG,
            font=("Segoe UI", 10),
            wraplength=760,
            justify="left",
        ).pack(anchor="w")

        section_title("Ako má AI vysvetľovať tréningy?")
        for label, value in [("Stručne", "short"), ("Normálne", "normal"), ("Podrobne", "detailed")]:
            tk.Radiobutton(
                card,
                text=label,
                variable=explain_var,
                value=value,
                fg=TEXT,
                bg=CARD_BG,
                selectcolor=CARD_BG,
                activebackground=CARD_BG,
                activeforeground=TEXT,
                font=("Segoe UI", 11),
            ).pack(anchor="w")

        def save_settings():
            payload = {
                "training_style": style_var.get(),
                "be_cautious_on_absence": cautious_var.get(),
                "no_volume_increase_on_absence": no_volume_var.get(),
                "gradual_return": gradual_var.get(),
                "allow_hard_sessions": hard_var.get(),
                "use_research": research_var.get(),
                "ai_explanation_level": explain_var.get(),
            }
            res = self._api_post("/api/coach/settings", payload)
            if res is not None:
                messagebox.showinfo("Uložené", "Nastavenia sú uložené. Zmeny platia od ďalšieho tréningu.")

        tk.Button(
            card,
            text="💾 Uložiť nastavenia",
            fg=TEXT,
            bg=ACCENT,
            activebackground=ACCENT,
            activeforeground=TEXT,
            font=("Segoe UI", 12, "bold"),
            relief="flat",
            padx=20,
            pady=8,
            command=save_settings,
        ).pack(anchor="w", pady=(18, 0))

    def temp_api_panel(self, name):
        self._api_clear()
        tk.Label(
            self.api_dark_area,
            text=name,
            fg=TEXT,
            bg=BG,
            font=("Segoe UI", 24),
        ).pack(pady=40)

    # ---------- Thread helpers ----------
    def refresh_list_threaded(self):
        threading.Thread(target=self.refresh_list, daemon=True).start()

    def add_training_threaded(self):
        threading.Thread(target=self.add_training_action, daemon=True).start()

    def delete_training_threaded(self):
        threading.Thread(target=self.delete_training_action, daemon=True).start()

    def load_trainees_threaded(self):
        threading.Thread(target=self.load_trainees, daemon=True).start()

    def auto_refresh_threaded(self):
        self.refresh_list_threaded()
        self.root.after(5000, self.auto_refresh_threaded)

    # ---------- Helpers ----------
    def logout_action(self):
        global token, role, current_user
        token = role = current_user = None
        self.main_frame.pack_forget()
        self.login_frame.pack(expand=True, fill="both")
        self.selected_training_id = None
        self.set_crud_buttons_state("disabled")
        if hasattr(self, "add_trainee_btn"):
            self.add_trainee_btn.pack_forget()
        self.close_popup()
        self._close_pb_window()
        self._close_dark_ui()
        self._close_api_dark_ui()

    def get_target_username(self):
        if role == "coach":
            v = self.selected_trainee.get().strip()
            if v:
                return v
            if getattr(self, "trainees", None):
                return self.trainees[0]
            return current_user
        return current_user

    def _set_selected_calendar_date(self, value):
        try:
            if isinstance(value, str):
                dt = datetime.strptime(value, "%Y-%m-%d").date()
            elif isinstance(value, datetime):
                dt = value.date()
            else:
                dt = value
            self.selected_calendar_date = dt.strftime("%Y-%m-%d")
        except Exception:
            self.selected_calendar_date = None

    def _is_light_color(self, color: str) -> bool:
        try:
            c = color.lstrip("#")
            if len(c) == 6:
                r, g, b = int(c[0:2], 16), int(c[2:4], 16), int(c[4:6], 16)
                luminance = (0.299 * r + 0.587 * g + 0.114 * b)
                return luminance > 186
        except Exception:
            pass
        return False

    def _weekly_volume_between(self, start_date, end_date):
        print(f"[weekly] _weekly_volume_between start={start_date} end={end_date}")
        daily = []
        for i in range(7):
            day = start_date + timedelta(days=i)
            day_trainings = []
            for t in self.all_trainings:
                tdate = self._normalize_date(t.get("date"))
                if tdate and tdate == day:
                    day_trainings.append(t)
            total_volume = 0
            for t in day_trainings:
                plan = t.get("training_plan", {}) or {}
                total_volume += int(plan.get("total_distance_m") or 0)
            daily.append((day, total_volume, len(day_trainings)))
            print(f"[weekly] day={day} volume={total_volume} count={len(day_trainings)}")
        return daily

    def _normalize_date(self, value):
        if isinstance(value, datetime):
            return value.date()
        if isinstance(value, date):
            return value
        if isinstance(value, str):
            try:
                return datetime.strptime(value, "%Y-%m-%d").date()
            except Exception:
                return None
        return None

    def _weekly_anchor_date(self):
        if self.selected_calendar_date:
            try:
                return datetime.strptime(self.selected_calendar_date, "%Y-%m-%d").date()
            except Exception:
                pass
        if not self.all_trainings:
            return None
        try:
            visible_month = self.main_cal.get_displayed_month()  # returns (year, month)
            visible_year, visible_mon = visible_month
        except Exception:
            visible_year, visible_mon = None, None

        month_trainings = []
        for tr in self.all_trainings:
            d = tr.get("date")
            try:
                dt = datetime.strptime(d, "%Y-%m-%d").date()
            except Exception:
                continue
            if visible_year and visible_mon:
                if dt.year == visible_year and dt.month == visible_mon:
                    month_trainings.append(dt)
            else:
                month_trainings.append(dt)
        if not month_trainings:
            return None
        today = datetime.now().date()
        closest = min(month_trainings, key=lambda d: abs((d - today).days))
        return closest

    def _check_horizon_ui(self, date_str: str) -> bool:
        try:
            dt = datetime.strptime(date_str, "%Y-%m-%d").date()
        except Exception:
            self.ui_msg("warn", "Validation", "Invalid date format.")
            return False
        if dt > date.today() + timedelta(days=DEMO_HORIZON_DAYS):
            self.ui_msg("warn", "Demo Limit", f"Planning horizon is {DEMO_HORIZON_DAYS} days.")
            return False
        return True

    def build_training_payload(self):
        date_value = self.cal.get_date().strftime("%Y-%m-%d")
        if not self._check_horizon_ui(date_value):
            return None
        time_value = self.time_entry.get().strip()
        desc_value = self.desc_text.get("1.0", "end").strip()
        activity = self.selected_activity.get()
        if not time_value:
            self.ui_msg("warn", "Validation", "Time is required (HH:MM).")
            return None
        try:
            datetime.strptime(time_value, "%H:%M")
        except ValueError:
            self.ui_msg("warn", "Validation", "Time format must be HH:MM.")
            return None
        if not desc_value:
            self.ui_msg("warn", "Validation", "Description is required.")
            return None
        return {"date": date_value, "time": time_value, "sport": activity, "description": desc_value}

    # ---------- Generate Training (AI-assisted) ----------
    def open_generate_training_window(self):
        if role != "coach":
            self.ui_msg("warn", "Access denied", "Only coaches can generate trainings.")
            return

        win = tk.Toplevel(self.root)
        win.title("Generate with AI")
        win.geometry("520x520")
        win.configure(bg="white")

        tk.Label(win, text="Selected date:", bg="white").pack(anchor="w", padx=20, pady=(20, 4))
        date_label = tk.Label(win, text=self.selected_calendar_date or "None", bg="white", fg="black")
        date_label.pack(anchor="w", padx=20)

        tk.Label(win, text="Coach Brief", bg="white").pack(anchor="w", padx=20, pady=(12, 4))
        placeholder = (
            "Describe what you want this week.\n"
            "Example: 1 tvrdý tréning, inak ľahko, zameraj sa na techniku, na celý týždeň."
        )
        brief_box = tk.Text(win, height=6, wrap="word", bg="#121212", fg="#EDEDED", insertbackground="#EDEDED", font=("Consolas", 11))
        brief_box.insert("1.0", placeholder)
        brief_box.configure(fg="#9A9A9A")
        def _clear_placeholder(event=None):
            if brief_box.get("1.0", "end").strip() == placeholder:
                brief_box.delete("1.0", "end")
                brief_box.configure(fg="#EDEDED")
        def _restore_placeholder(event=None):
            if not brief_box.get("1.0", "end").strip():
                brief_box.configure(fg="#9A9A9A")
                brief_box.insert("1.0", placeholder)
        brief_box.bind("<FocusIn>", _clear_placeholder)
        brief_box.bind("<FocusOut>", _restore_placeholder)
        brief_box.pack(fill="both", expand=False, padx=20, pady=(0, 10))

        result_box = tk.Text(win, height=12, wrap="word", bg="#111111", fg="#e0e0e0")
        result_box.pack(fill="both", expand=True, padx=20, pady=15)
        last_training_id = {"id": None, "owner": None}

        btn_frame = tk.Frame(win, bg="white")
        btn_frame.pack(fill="x", padx=20, pady=(0, 12))

        def submit():
            if not self.selected_calendar_date:
                self.ui_msg("warn", "No date selected", "Please select a date on the main calendar.")
                return
            if not self._check_horizon_ui(self.selected_calendar_date):
                return
            coach_brief_text = brief_box.get("1.0", "end").strip()
            if coach_brief_text == placeholder.strip():
                coach_brief_text = ""
            payload = {
                "date": self.selected_calendar_date,
                "coach_brief_text": coach_brief_text,
            }
            owner = self.get_target_username()
            approve_btn.configure(state="disabled")
            reject_btn.configure(state="disabled")

            def run():
                try:
                    endpoint = f"training/generate?owner={owner}" if owner else "training/generate"
                    res = api_request("POST", endpoint, payload)
                    pretty = json.dumps(res, indent=2, ensure_ascii=False)
                    def update_ui():
                        result_box.delete("1.0", "end")
                        result_box.insert("1.0", pretty)
                        last_training_id["id"] = res.get("id")
                        last_training_id["owner"] = owner
                        if last_training_id["id"]:
                            approve_btn.configure(state="normal")
                            reject_btn.configure(state="normal")
                    self.ui(update_ui)
                    self.refresh_list_threaded()
                except Exception as e:
                    self.ui_msg("error", "Generate failed", str(e))

            threading.Thread(target=run, daemon=True).start()

        ttk.Button(btn_frame, text="Generate", style="Blue.TButton", command=submit).pack(side="left", expand=True, fill="x", padx=(0, 6))

        def set_status(status: str):
            tid = last_training_id["id"]
            owner = last_training_id["owner"] or self.get_target_username()
            if not tid:
                self.ui_msg("warn", "Approval", "Najprv vygeneruj tréning.")
                return
            approve_btn.configure(state="disabled")
            reject_btn.configure(state="disabled")
            def run():
                try:
                    endpoint = f"trainings/{owner}/{tid}/{status}"
                    api_request("POST", endpoint, {})
                    self.refresh_list_threaded()
                    self.ui_msg("info", "Status", f"Tréning označený ako {status}.")
                except Exception as e:
                    self.ui_msg("error", "Status", str(e))
                finally:
                    self.ui(lambda: (approve_btn.configure(state="normal"), reject_btn.configure(state="normal")))
            threading.Thread(target=run, daemon=True).start()

        approve_btn = ttk.Button(btn_frame, text="Schváliť", style="Green.TButton", command=lambda: set_status("approve"))
        approve_btn.pack(side="left", expand=True, fill="x", padx=(6, 6))
        reject_btn = ttk.Button(btn_frame, text="Zamietnuť", style="Red.TButton", command=lambda: set_status("reject"))
        reject_btn.pack(side="left", expand=True, fill="x", padx=(6, 0))
        approve_btn.configure(state="disabled")
        reject_btn.configure(state="disabled")

    # ---------- PB Improvement Predictor ----------
    def open_pb_predictor_window(self):
        if self.pb_window and self.pb_window.winfo_exists():
            self.pb_window.lift()
            return

        self.pb_window = tk.Toplevel(self.root)
        self.pb_window.title("PB Improvement Predictor")
        self.pb_window.geometry("420x330")
        self.pb_window.configure(bg="white")
        self.pb_window.protocol("WM_DELETE_WINDOW", self._close_pb_window)

        frame = tk.Frame(self.pb_window, bg="white", padx=20, pady=20)
        frame.pack(fill="both", expand=True)
        frame.columnconfigure(1, weight=1)

        self.pb_username_var = tk.StringVar(value=self.get_target_username() or "")
        self.pb_pb_var = tk.StringVar(value="00:28.50")
        self.pb_days_var = tk.StringVar(value="5")
        self.pb_result_var = tk.StringVar(value="Výsledok: ...")
        self.pb_last_score = None
        self.pb_last_breakdown = None

        tk.Label(frame, text="Username:", bg="white").grid(row=0, column=0, sticky="w", pady=5)
        username_entry = ttk.Entry(frame, textvariable=self.pb_username_var)
        username_entry.grid(row=0, column=1, sticky="ew", pady=5)

        tk.Label(frame, text="PB čas (mm:ss.xx):", bg="white").grid(row=1, column=0, sticky="w", pady=5)
        pb_entry = ttk.Entry(frame, textvariable=self.pb_pb_var)
        pb_entry.grid(row=1, column=1, sticky="ew", pady=5)

        tk.Label(frame, text="Dni do súťaže:", bg="white").grid(row=2, column=0, sticky="w", pady=5)
        days_entry = ttk.Entry(frame, textvariable=self.pb_days_var)
        days_entry.grid(row=2, column=1, sticky="ew", pady=5)

        calc_btn = ttk.Button(frame, text="Vypočítať PB Improvement")
        calc_btn.grid(row=3, column=0, columnspan=2, pady=(12, 8))

        self.pb_detail_btn = ttk.Button(
            frame,
            text="Zobraziť výpočet",
            state="disabled",
            command=self._show_pb_breakdown,
        )
        self.pb_detail_btn.grid(row=4, column=0, columnspan=2, pady=(0, 8))

        tk.Label(frame, textvariable=self.pb_result_var, font=("Arial", 13, "bold"), bg="white").grid(
            row=5, column=0, columnspan=2, pady=(10, 0)
        )

        def on_calculate():
            username = self.pb_username_var.get().strip()
            if not username:
                self.ui_msg("warn", "PB Predictor", "Zadaj používateľa.")
                return

            pb_input = self.pb_pb_var.get().strip().replace(",", ".")
            pb_seconds = time_to_seconds(pb_input)
            if pb_seconds is None:
                self.ui_msg("error", "PB Predictor", "Nesprávny formát PB času.")
                return
            try:
                days = int(self.pb_days_var.get())
            except ValueError:
                self.ui_msg("error", "PB Predictor", "Dni do súťaže musí byť číslo.")
                return

            self.pb_result_var.set("Počítam ...")
            self.pb_last_score = None
            self.pb_last_breakdown = None
            calc_btn.configure(state="disabled")
            if self.pb_detail_btn:
                self.pb_detail_btn.configure(state="disabled")
            threading.Thread(
                target=self._compute_pb_prediction,
                args=(username, pb_seconds, days, self.pb_result_var, calc_btn, self.pb_detail_btn),
                daemon=True,
            ).start()

        calc_btn.configure(command=on_calculate)
        username_entry.focus_set()

    def _close_pb_window(self):
        if self.pb_window and self.pb_window.winfo_exists():
            self.pb_window.destroy()
        self.pb_window = None
        self.pb_username_var = None
        self.pb_pb_var = None
        self.pb_days_var = None
        self.pb_result_var = None
        self.pb_detail_btn = None
        self.pb_last_score = None
        self.pb_last_breakdown = None

    def _compute_pb_prediction(self, username, pb_seconds, days, result_var, calc_button, detail_button):
        try:
            trainings = api_request("GET", f"trainings/{username}")
        except Exception as e:
            self.ui_msg("error", "PB Predictor", f"API error: {e}")
            def reset():
                if calc_button.winfo_exists():
                    calc_button.configure(state="normal")
                try:
                    result_var.set("Výsledok: ...")
                except tk.TclError:
                    pass
                if detail_button and detail_button.winfo_exists():
                    detail_button.configure(state="disabled")
            self.ui(reset)
            return

        trainings = sorted(trainings, key=lambda t: (t.get("date", ""), t.get("time", "")))
        parsed = []
        for t in trainings:
            desc = t.get("description", "") or ""
            volume = extract_volume(desc)
            quality = extract_quality(desc)

            set_time = None
            for key in ("time_of_set", "set_time"):
                set_val = t.get(key)
                if not set_val:
                    continue
                set_seconds = time_to_seconds(str(set_val).replace(",", "."))
                if set_seconds is not None:
                    set_time = set_seconds
                    break

            delta = set_time - pb_seconds if set_time is not None else 0.5
            parsed.append({
                "volume": volume,
                "quality": quality,
                "delta_pb": delta,
                "fatigue": 0.75
            })

        if not parsed:
            self.ui_msg("warn", "PB Predictor", f"Žiadne tréningy pre {username}.")
        elif len(parsed) < 3:
            self.ui_msg("warn", "PB Predictor", "Potrebujeme aspoň 3 tréningy, výsledok môže byť nepresný.")

        score, breakdown = pb_improvement_details(parsed, days, pb_seconds)

        def apply():
            try:
                result_var.set(f"Pravdepodobnosť zlepšenia PB: {score:.2f}%")
                self.pb_last_score = score
                self.pb_last_breakdown = breakdown
            except tk.TclError:
                return
            if calc_button.winfo_exists():
                calc_button.configure(state="normal")
            if detail_button and detail_button.winfo_exists():
                detail_button.configure(state="normal")
        self.ui(apply)

    def _show_pb_breakdown(self):
        """Show a quick breakdown of the PB improvement calculation."""
        if not self.pb_last_breakdown:
            messagebox.showinfo("PB Predictor", "Najprv vypočítaj PB improvement.")
            return

        bd = self.pb_last_breakdown
        form_pct = bd["form"] * 100
        trend_pct = bd["trend"] * 100
        recovery_pct = bd["recovery"] * 100
        dopt_pct = bd["dopt"] * 100

        msg = (
            f"Forma (váha 35 %): {form_pct:.1f} % → príspevok {(bd['form'] * 0.35 * 100):.1f} %\n"
            f"Trend (váha 30 %): {trend_pct:.1f} % → príspevok {(bd['trend'] * 0.30 * 100):.1f} %\n"
            f"Recovery (váha 20 %): {recovery_pct:.1f} % → príspevok {(bd['recovery'] * 0.20 * 100):.1f} %\n"
            f"Optimálne dni (váha 15 %): {dopt_pct:.1f} % → príspevok {(bd['dopt'] * 0.15 * 100):.1f} %\n"
            f"\nCelkové skóre: {bd['weighted_score']:.2f} %"
        )
        messagebox.showinfo("Výpočet PB Improvement", msg)

    # ---------- Coach: trainees ----------
    def load_trainees(self):
        try:
            data = api_request("GET", "trainees")
        except Exception as e:
            self.ui_msg("error", "Load Trainees", str(e))
            return
        def apply():
            self.trainees = data or []
            if self.trainees:
                self.coach_top.pack(fill="x", padx=12, pady=(10, 0))
                self.user_select.configure(values=self.trainees)
                if not self.selected_trainee.get():
                    self.selected_trainee.set(self.trainees[0])
            else:
                self.coach_top.pack_forget()
            self.refresh_list_threaded()
        self.ui(apply)

    def on_trainee_changed(self):
        self.close_popup()
        self.refresh_list_threaded()

    def open_add_trainee_window(self):
        self.ui_msg("info", "Demo", "Adding athletes is disabled in the demo.")
        return
        if role != "coach":
            messagebox.showinfo("Access Denied", "Only coaches can add trainees.")
            return

        win = tk.Toplevel(self.root)
        win.title("Add Trainee")
        win.geometry("320x220")
        win.configure(bg="white")
        win.transient(self.root)

        tk.Label(win, text="New Trainee", bg="white", font=("Arial", 13, "bold")).pack(pady=10)

        form = tk.Frame(win, bg="white")
        form.pack(fill="both", expand=True, padx=20, pady=5)

        tk.Label(form, text="Username:", bg="white").grid(row=0, column=0, sticky="w", pady=4)
        username_entry = ttk.Entry(form)
        username_entry.grid(row=0, column=1, sticky="ew", pady=4)

        tk.Label(form, text="Password:", bg="white").grid(row=1, column=0, sticky="w", pady=4)
        password_entry = ttk.Entry(form, show="*")
        password_entry.grid(row=1, column=1, sticky="ew", pady=4)

        tk.Label(form, text="Confirm Password:", bg="white").grid(row=2, column=0, sticky="w", pady=4)
        confirm_entry = ttk.Entry(form, show="*")
        confirm_entry.grid(row=2, column=1, sticky="ew", pady=4)

        form.columnconfigure(1, weight=1)

        def submit():
            username = username_entry.get().strip()
            password = password_entry.get().strip()
            confirm = confirm_entry.get().strip()
            if not username or not password:
                messagebox.showwarning("Validation", "Username and password are required.")
                return
            if password != confirm:
                messagebox.showwarning("Validation", "Passwords do not match.")
                return

            submit_btn.config(state="disabled")

            def run():
                try:
                    api_request("POST", "trainees", {"username": username, "password": password})
                    self.ui_msg("info", "Trainee Added", f"{username} created successfully.")
                    self.load_trainees_threaded()
                    self.ui(win.destroy)
                except Exception as e:
                    self.ui_msg("error", "Add Trainee", str(e))
                    self.ui(lambda: submit_btn.config(state="normal"))

            threading.Thread(target=run, daemon=True).start()

        actions = tk.Frame(win, bg="white")
        actions.pack(pady=10)
        submit_btn = ttk.Button(actions, text="Create Trainee", command=submit)
        submit_btn.pack(side="left", padx=5)
        ttk.Button(actions, text="Cancel", command=win.destroy).pack(side="left", padx=5)

    # ---------- Server sync (thread target) ----------
    def refresh_list(self):
        if not token:
            return
        username = self.get_target_username()
        try:
            data = api_request("GET", f"trainings/{username}")
        except DemoError as e:
            self.ui_msg("error", "Error", str(e))
            return
        except Exception as e:
            self.ui_msg("error", "Error", str(e))
            return

        def apply():
            self.all_trainings = data or []
            self.update_calendar_marks()
            self.refresh_dark_view()
            self.set_crud_buttons_state("normal" if role == "coach" else "disabled")
        self.ui(apply)

    def show_weekly_graph(self):
        """Display a bar chart for the week anchored to calendar selection or visible trainings."""
        print("[weekly] show_weekly_graph called")
        if not token:
            self.ui_msg("warn", "Týždenný graf", "Najprv sa prihlás a načítaj tréningy.")
            return

        anchor_date = self._weekly_anchor_date()
        print(f"[weekly] anchor_date={anchor_date}")
        if not anchor_date:
            self.ui_msg("info", "Týždenný graf", "No trainings in this month.")
            return

        end_date = anchor_date
        start_date = end_date - timedelta(days=6)

        daily = self._weekly_volume_between(start_date, end_date)
        print(f"[weekly] daily_records={daily}")

        if all(count == 0 for _, _, count in daily):
            self.ui_msg("info", "Týždenný graf", "V tomto týždni nemáme žiadne tréningy. Stlač Refresh, ak si niečo pridal.")
            return

        max_volume = max(v for _, v, _ in daily) or 1
        win = tk.Toplevel(self.root)
        win.title("Týždenný graf tréningov (objem)")
        win.geometry("640x420")
        win.configure(bg="white")

        tk.Label(
            win,
            text=f"Objem za týždeň (od {start_date.strftime('%d.%m.')} do {end_date.strftime('%d.%m.')})",
            bg="white",
            font=("Arial", 12, "bold")
        ).pack(pady=(10, 6))

        canvas = tk.Canvas(win, width=600, height=300, bg="#f8f9fa", highlightthickness=0)
        canvas.pack(padx=14, pady=10, fill="x")

        margin_left, margin_bottom, margin_top = 50, 40, 20
        chart_w = 600 - margin_left - 20
        chart_h = 300 - margin_bottom - margin_top
        bar_w = chart_w / 7 * 0.6
        bar_gap = (chart_w / 7) - bar_w
        base_x = margin_left
        base_y = margin_top + chart_h

        canvas.create_line(margin_left, margin_top, margin_left, base_y, fill="#444", width=1.5)
        canvas.create_line(margin_left, base_y, margin_left + chart_w, base_y, fill="#444", width=1.5)

        for idx, (day, volume, count) in enumerate(daily):
            x0 = base_x + idx * (bar_w + bar_gap)
            x1 = x0 + bar_w
            bar_height = (volume / max_volume) * chart_h
            y1 = base_y - bar_height
            fill = "#3498db" if volume > 0 else "#d6e9f7"
            canvas.create_rectangle(x0, y1, x1, base_y, fill=fill, outline="#2c81b5")
            canvas.create_text((x0 + x1) / 2, y1 - 10, text=f"{volume} m", fill="#333", font=("Arial", 9))
            canvas.create_text((x0 + x1) / 2, base_y + 12, text=day.strftime("%a"), fill="#333", font=("Arial", 9))
            canvas.create_text((x0 + x1) / 2, base_y + 26, text=f"{count} tréningov", fill="#777", font=("Arial", 8))

        for v in (0, max_volume / 2, max_volume):
            y = base_y - (v / max_volume) * chart_h
            canvas.create_line(margin_left - 5, y, margin_left, y, fill="#444")
            canvas.create_text(margin_left - 10, y, text=f"{int(v)}", anchor="e", fill="#444", font=("Arial", 9))

    def update_calendar_marks(self):
        """Highlight days and attach per-training tooltips with descriptions."""
        for ev_id in self.main_cal.get_calevents():
            self.main_cal.calevent_remove(ev_id)
        color_tags = {}

        def tag_for_color(color: str) -> str:
            if color in color_tags:
                return color_tags[color]
            tag_name = f"c_{len(color_tags)}"
            fg = "#000000" if self._is_light_color(color) else "#ffffff"
            self.main_cal.tag_config(tag_name, background=color, foreground=fg)
            color_tags[color] = tag_name
            return tag_name

        for tr in self.all_trainings:
            d = tr.get("date")
            if not d:
                continue
            try:
                dt = datetime.strptime(d, "%Y-%m-%d")
            except ValueError:
                continue
            plan = tr.get("training_plan", {})
            title = plan.get("title") or tr.get("title") or "Training"
            summary = plan.get("summary", {})
            main_goal = summary.get("main_goal") or tr.get("description", "")
            lines = plan.get("lines", [])
            training_type = tr.get("training_type") or plan.get("training_type")
            color = tr.get("color") or TRAINING_TYPE_COLORS.get(training_type, "#d1ecf1")
            tip_parts = [title]
            if main_goal:
                tip_parts.append(main_goal)
            tip_parts.extend(lines[:5])
            tip = "\n".join([p for p in tip_parts if p])
            tags = [tag_for_color(color)]
            self.main_cal.calevent_create(dt, tip, tags)

    # ---------- Calendar click ----------
    def on_main_calendar_select(self, _event=None):
        selected = self.main_cal.get_date()  # str or date depending on platform/config
        self.cal.set_date(selected)  # keep add-form in sync
        self._set_selected_calendar_date(selected)
        self.open_day_popup(selected)

    # ---------- Popup lifecycle ----------
    def close_popup(self):
        """Safely close popup and release any grabs."""
        try:
            if self.popup and tk.Toplevel.winfo_exists(self.popup):
                try:
                    self.popup.grab_release()
                except Exception:
                    pass
                self.popup.destroy()
        except Exception:
            pass
        finally:
            self.popup = None
            self.popup_listbox = None
            self.popup_id_by_index = {}
            self.selected_training_id = None

    def open_day_popup(self, dt):
        """Popup listing trainings for the clicked date, with Edit/Delete and Quick-Add."""
        if isinstance(dt, str):
            try:
                dt = datetime.strptime(dt, "%Y-%m-%d").date()
            except Exception:
                self.ui_msg("error", "Error", f"Invalid date format: {dt}")
                return

        self._set_selected_calendar_date(dt)
        date_str = dt.strftime("%Y-%m-%d")
        rows = [tr for tr in self.all_trainings if tr.get("date") == date_str]
        rows = sorted(rows, key=lambda x: x.get("time", ""))

        self.close_popup()

        win = tk.Toplevel(self.root)
        self.popup = win
        win.title(f"Trainings on {date_str}")
        win.geometry("560x420")
        win.configure(bg="white")
        win.transient(self.root)
        win.focus_set()
        win.protocol("WM_DELETE_WINDOW", self.close_popup)

        header = tk.Frame(win, bg="white")
        header.pack(fill="x", padx=12, pady=(10, 0))
        tk.Label(header, text=f"Trainings on {date_str}", bg="white", font=("Arial", 13, "bold")).pack(side="left")

        list_frame = tk.Frame(win, bg="white")
        list_frame.pack(fill="both", expand=True, padx=12, pady=10)

        scrollbar = ttk.Scrollbar(list_frame, orient="vertical")
        listbox = tk.Listbox(list_frame, height=14, yscrollcommand=scrollbar.set)
        scrollbar.config(command=listbox.yview)
        listbox.pack(side="left", fill="both", expand=True)
        scrollbar.pack(side="right", fill="y")

        self.popup_listbox = listbox
        self.popup_id_by_index = {}
        self.training_cache.clear()

        actionable_indices = []
        for i, tr in enumerate(rows):
            tid = tr.get("id")
            time_s = tr.get('time', '--:--')
            sport_s = tr.get('sport', '—')
            desc_s = (tr.get('description', '') or '').strip()
            label = f"{time_s}  |  {sport_s}  |  {desc_s}"
            listbox.insert("end", label)
            if tid:
                self.training_cache[tid] = tr
                self.popup_id_by_index[i] = tid
                actionable_indices.append(i)

        detail_frame = tk.Frame(win, bg="white")
        detail_frame.pack(fill="both", expand=True, padx=12, pady=(0, 10))
        detail_text = tk.Text(
            detail_frame,
            height=10,
            wrap="word",
            bg="#0f0f0f",
            fg="#e6e6e6",
            insertbackground="#ffffff",
            font=("Courier New", 11),
        )
        detail_text.pack(fill="both", expand=True)

        def render_detail(tr_id: str | None):
            detail_text.configure(state="normal")
            detail_text.delete("1.0", "end")
            if not tr_id:
                detail_text.insert("1.0", "Select a training to view details.")
                detail_text.configure(state="disabled")
                return
            tr = self.training_cache.get(tr_id, {})
            plan = tr.get("training_plan", {})
            training_text = tr.get("training_text") or plan.get("training_text")
            if training_text:
                detail_text.insert("1.0", str(training_text))
            else:
                lines = plan.get("lines")
                if lines and isinstance(lines, list):
                    for line in lines:
                        detail_text.insert("end", f"{line}\n")
                else:
                    desc = (tr.get("description", "") or "").strip()
                    detail_text.insert("1.0", desc if desc else "No details available.")
            detail_text.configure(state="disabled")

        def on_select(_evt=None):
            sel = listbox.curselection()
            self.selected_training_id = self.popup_id_by_index.get(sel[0]) if sel else None
            self.set_crud_buttons_state("normal" if role == "coach" else "disabled")
            render_detail(self.selected_training_id)
        listbox.bind("<<ListboxSelect>>", on_select)

        if len(actionable_indices) >= 1:
            first_idx = actionable_indices[0]
            listbox.selection_set(first_idx)
            listbox.activate(first_idx)
            listbox.see(first_idx)
            self.selected_training_id = self.popup_id_by_index.get(first_idx)
            render_detail(self.selected_training_id)
        else:
            render_detail(None)

        btns = tk.Frame(win, bg="white")
        btns.pack(fill="x", padx=12, pady=(0, 12))

        def do_edit():
            if role != "coach":
                messagebox.showinfo("Not Allowed", "Only coaches can edit trainings.")
                return
            if not self.selected_training_id:
                messagebox.showwarning("Select", "Pick a training with a valid ID to edit.")
                return
            self.open_edit_window()

        def do_delete():
            if role != "coach":
                messagebox.showinfo("Not Allowed", "Only coaches can delete trainings.")
                return
            if not self.selected_training_id:
                messagebox.showwarning("Select", "Pick a training with a valid ID to delete.")
                return
            if not messagebox.askyesno("Confirm", "Delete selected training?"):
                return
            tr = self.training_cache.get(self.selected_training_id, {})
            owner = tr.get("username") or tr.get("owner") or self.get_target_username()
            def _del():
                try:
                    api_request("DELETE", f"trainings/{owner}/{self.selected_training_id}")
                    self.ui_msg("info", "Deleted", "Training deleted.")
                    self.selected_training_id = None
                    self.refresh_list_threaded()
                    self.ui(self.open_day_popup, date_str)
                except Exception as e:
                    self.ui_msg("error", "Error", str(e))
            threading.Thread(target=_del, daemon=True).start()

        def do_quick_add():
            self.cal.set_date(dt)
            self.time_entry.focus_set()
            messagebox.showinfo("Quick Add", "Date prefilled on the left. Enter time/sport/description, then click Add.")

        ttk.Button(btns, text="Quick-Add for this date", style="Green.TButton", command=do_quick_add).pack(side="left", padx=4)
        ttk.Button(btns, text="Edit", style="Purple.TButton", command=do_edit).pack(side="left", padx=4)
        ttk.Button(btns, text="Delete", style="Red.TButton", command=do_delete).pack(side="left", padx=4)
        ttk.Button(btns, text="Close", command=self.close_popup).pack(side="right", padx=4)

        listbox.bind("<Double-Button-1>", lambda _e: do_edit())
        self.set_crud_buttons_state("normal" if role == "coach" else "disabled")

    # ---------- CRUD (thread targets) ----------
    def add_training_action(self):
        if role != "coach":
            self.ui_msg("info", "Not Allowed", "Only coaches can add trainings.")
            return
        payload = self.build_training_payload()
        if not payload:
            return
        username = self.get_target_username()
        try:
            api_request("POST", f"trainings/{username}", data=payload)
            self.ui_msg("info", "Added", "Training added successfully!")
            selected = self.main_cal.get_date()
            self.refresh_list_threaded()
            self.ui(self.open_day_popup, selected)
        except Exception as e:
            self.ui_msg("error", "Error", str(e))

    def open_edit_window(self):
        if role != "coach":
            messagebox.showinfo("Not Allowed", "Only coaches can edit trainings.")
            return
        if not self.selected_training_id:
            messagebox.showwarning("Select", "Please select a training (click a date, choose one).")
            return
        training = self.training_cache.get(self.selected_training_id)
        if not training:
            messagebox.showerror("Error", "Training not found.")
            return

        edit_win = tk.Toplevel(self.root)
        edit_win.title("Edit Training")
        edit_win.geometry("420x360")
        edit_win.configure(bg="white")
        edit_win.transient(self.root)
        edit_win.focus_set()

        tk.Label(edit_win, text="Edit Training", bg="white", font=("Arial", 14, "bold")).pack(pady=10)
        date_entry = DateEntry(edit_win, date_pattern="yyyy-mm-dd")
        date_entry.set_date(datetime.strptime(training["date"], "%Y-%m-%d"))
        date_entry.pack(pady=5)

        time_entry = ttk.Entry(edit_win)
        time_entry.insert(0, training.get("time", ""))
        time_entry.pack(pady=5)

        sport_combo = ttk.Combobox(edit_win, values=self.activities, state="readonly")
        sport_combo.set(training.get("sport", self.activities[0]))
        sport_combo.pack(pady=5)

        desc_frame = tk.Frame(edit_win, bg="white")
        desc_frame.pack(pady=5, fill="both", expand=True)
        desc_frame.configure(height=200, highlightbackground="#dcdcdc", highlightthickness=1)
        desc_frame.pack_propagate(False)
        desc_entry = tk.Text(
            desc_frame,
            height=10,
            width=40,
            wrap="word",
            bg="#000000",
            fg="#ffffff",
            insertbackground="#ffffff",
        )
        desc_scroll = ttk.Scrollbar(desc_frame, orient="vertical", command=desc_entry.yview)
        desc_entry.configure(yscrollcommand=desc_scroll.set)
        desc_entry.insert("1.0", training.get("description", ""))
        desc_entry.pack(side="left", fill="both", expand=True)
        desc_scroll.pack(side="right", fill="y")

        def save():
            date_value = date_entry.get_date().strftime("%Y-%m-%d")
            if not self._check_horizon_ui(date_value):
                return
            payload = {
                "date": date_value,
                "time": time_entry.get().strip(),
                "sport": sport_combo.get(),
                "description": desc_entry.get("1.0", "end").strip()
            }
            if not payload["time"]:
                messagebox.showwarning("Validation", "Time is required.")
                return
            owner = training.get("username") or training.get("owner") or self.get_target_username()
            def _put():
                try:
                    api_request("PUT", f"trainings/{owner}/{self.selected_training_id}", data=payload)
                    self.ui_msg("info", "Updated", "Training updated successfully!")
                    self.ui(edit_win.destroy)
                    selected = self.main_cal.get_date()
                    self.refresh_list_threaded()
                    self.ui(self.open_day_popup, selected)
                except Exception as e:
                    self.ui_msg("error", "Error", str(e))
            threading.Thread(target=_put, daemon=True).start()

        ttk.Button(edit_win, text="Save", style="Green.TButton", command=save).pack(pady=10)

    def _fallback_first_training_on_selected_date(self):
        date_val = self.main_cal.get_date()
        if isinstance(date_val, str):
            try:
                date_val = datetime.strptime(date_val, "%Y-%m-%d").date()
            except Exception:
                return False
        date_str = date_val.strftime("%Y-%m-%d")
        rows = [tr for tr in self.all_trainings if tr.get("date") == date_str]
        rows = sorted(rows, key=lambda x: x.get("time", ""))
        for tr in rows:
            tid = tr.get("id")
            if tid:
                self.training_cache[tid] = tr
                self.selected_training_id = tid
                return True
        return False

    def delete_training_action(self):
        if role != "coach":
            self.ui_msg("info", "Not Allowed", "Only coaches can delete trainings.")
            return

        if not self.selected_training_id:
            picked = self._fallback_first_training_on_selected_date()
            if not picked:
                self.ui_msg("warn", "Select", "Click a date with trainings (the first one will be chosen automatically), or open the popup and select one.")
                return

        if not messagebox.askyesno("Confirm", "Delete selected training?"):
            return

        tr = self.training_cache.get(self.selected_training_id, {})
        owner = tr.get("username") or tr.get("owner") or self.get_target_username()

        def _del():
            try:
                api_request("DELETE", f"trainings/{owner}/{self.selected_training_id}")
                self.ui_msg("info", "Deleted", "Training deleted.")
                self.selected_training_id = None
                selected = self.main_cal.get_date()
                self.refresh_list_threaded()
                self.ui(self.open_day_popup, selected)
            except Exception as e:
                self.ui_msg("error", "Error", str(e))
        threading.Thread(target=_del, daemon=True).start()

    # ---------- Profiles (coach) ----------
    def open_profile_window(self):
        if role == "coach":
            self._open_coach_profile_selector()
        else:
            if not current_user:
                messagebox.showerror("Unavailable", "No trainee is logged in.")
                return
            self.show_profile_popup(current_user, allow_import=False)

    def _open_coach_profile_selector(self):
        # Create popup for trainee selection
        select_win = tk.Toplevel(self.root)
        select_win.title("Select Trainee")
        select_win.geometry("400x200")
        select_win.configure(bg="white")

        tk.Label(select_win, text="Select a trainee:", bg="white", font=("Arial", 12)).pack(pady=10)

        try:
            trainees = api_request("GET", "trainees")
        except DemoError as e:
            messagebox.showerror("Error", f"Failed to fetch trainees:\n{e}")
            select_win.destroy()
            return

        combo = ttk.Combobox(select_win, values=trainees, state="readonly", width=30)
        combo.pack(pady=10)
        if trainees:
            combo.current(0)

        def open_selected_profile():
            selected = combo.get()
            if not selected:
                messagebox.showwarning("No selection", "Please select a trainee.")
                return
            select_win.destroy()
            self.show_profile_popup(selected)

        ttk.Button(select_win, text="Open Profile", command=open_selected_profile).pack(pady=10)
        ttk.Button(select_win, text="Cancel", command=select_win.destroy).pack()

    def show_profile_popup(self, username, allow_import=True):
        win = tk.Toplevel(self.root)
        win.title(f"{'Edit' if allow_import else 'My'} Profile — {username}")
        win.geometry("500x320")
        win.configure(bg="white")

        tk.Label(
            win,
            text=("Update" if allow_import else "Personal bests for") + f" {username}",
            bg="white",
            font=("Arial", 13, "bold")
        ).pack(pady=(10, 4))

        url_entry = None
        if allow_import:
            tk.Label(win, text="Paste SwimRankings URL:", bg="white").pack()
            url_entry = ttk.Entry(win, width=70)
            url_entry.pack(pady=8)
        else:
            tk.Label(win, text="Below are the imported/pasted PBs.", bg="white").pack(pady=(0, 8))

        if allow_import:
            def import_swimrankings_local():
                messagebox.showinfo("Demo", "SwimRankings import is disabled in the demo.")

            ttk.Button(
                win,
                text="Import from SwimRankings",
                command=import_swimrankings_local,
                state="disabled",
            ).pack(pady=(4, 10))

        columns = ("event", "time")
        tree = ttk.Treeview(win, columns=columns, show="headings", height=12)
        tree.heading("event", text="Discipline")
        tree.heading("time", text="Best Time")
        tree.column("event", width=340)
        tree.column("time", width=120, anchor="center")
        tree.pack(fill="both", expand=True, padx=10, pady=10)
        self.load_personal_bests(username, tree)

        ttk.Button(win, text="Close", command=win.destroy).pack(pady=(0, 6))

    def load_personal_bests(self, username, tree):
        try:
            res = api_request("GET", f"profile/{username}")
            pbs = res.get("personal_bests", {})
            for item in tree.get_children():
                tree.delete(item)
            for event, time in pbs.items():
                tree.insert("", "end", values=(event, time))
        except Exception as e:
            messagebox.showerror("Error", f"Failed to load PBs:\n{e}")

    def import_swimrankings(self, username, url_entry, tree):
        messagebox.showinfo("Demo", "SwimRankings import is disabled in the demo.")

if __name__ == "__main__":
    root = tk.Tk()
    app = TrainingApp(root)
    root.mainloop()
