from __future__ import annotations

import json
import re
import uuid
from dataclasses import dataclass
from datetime import date, datetime, timedelta
from pathlib import Path
from threading import Lock
from typing import Any, Dict, List, Optional

DATA_DIR = Path(__file__).resolve().parent / "demo_data"
USERS_FILE = DATA_DIR / "users.json"
ATHLETES_FILE = DATA_DIR / "athletes.json"
TRAININGS_FILE = DATA_DIR / "trainings.json"

DEMO_HORIZON_DAYS = 14
_LOCK = Lock()


class DemoError(RuntimeError):
    pass


@dataclass(frozen=True)
class DemoUser:
    username: str
    role: str
    display_name: str


def _load_json(path: Path, default: Dict[str, Any]) -> Dict[str, Any]:
    if not path.exists():
        return default
    return json.loads(path.read_text(encoding="utf-8"))


def _save_json(path: Path, data: Dict[str, Any]) -> None:
    path.parent.mkdir(parents=True, exist_ok=True)
    path.write_text(json.dumps(data, indent=2, ensure_ascii=False), encoding="utf-8")


def ensure_seed_data() -> None:
    with _LOCK:
        if USERS_FILE.exists() and ATHLETES_FILE.exists() and TRAININGS_FILE.exists():
            return
        DATA_DIR.mkdir(parents=True, exist_ok=True)
        users, athletes = _seed_users_athletes()
        trainings = _seed_trainings(athletes)
        _save_json(USERS_FILE, {"users": users})
        _save_json(ATHLETES_FILE, {"athletes": athletes})
        _save_json(TRAININGS_FILE, {"trainings": trainings})


def login(username: str, password: str) -> DemoUser:
    data = _load_json(USERS_FILE, {"users": []})
    for user in data.get("users", []):
        if user.get("username") == username and user.get("password") == password:
            return DemoUser(username=username, role=user.get("role", "trainee"), display_name=user.get("display_name", username))
    raise DemoError("Invalid credentials.")


def request(
    method: str,
    endpoint: str,
    data: Optional[Dict[str, Any]] = None,
    current_user: Optional[str] = None,
    role: Optional[str] = None,
    target_username: Optional[str] = None,
) -> Any:
    if not current_user or not role:
        raise DemoError("Not authenticated.")
    path, query = _split_endpoint(endpoint)
    parts = [p for p in path.split("/") if p]
    method = method.upper()

    if not parts:
        raise DemoError("Invalid endpoint.")

    if parts[0] == "trainees":
        if role != "coach":
            raise DemoError("Access denied.")
        if method == "GET":
            return _list_trainees(current_user)
        if method == "POST":
            raise DemoError("Adding athletes is disabled in the demo.")

    if parts[0] == "training" and len(parts) >= 2 and parts[1] == "generate":
        if role != "coach":
            raise DemoError("Only coaches can generate trainings.")
        owner = query.get("owner") or target_username or current_user
        _assert_access(owner, current_user, role)
        if method != "POST":
            raise DemoError("Invalid method.")
        payload = data or {}
        return _generate_training(owner, payload)

    if parts[0] == "trainings":
        if len(parts) == 2 and method == "GET":
            username = parts[1]
            _assert_access(username, current_user, role)
            return _list_trainings(username)
        if len(parts) == 2 and method == "POST":
            if role != "coach":
                raise DemoError("Athlete accounts are read-only.")
            username = parts[1]
            _assert_access(username, current_user, role)
            return _add_training(username, data or {})
        if len(parts) == 3 and method == "PUT":
            if role != "coach":
                raise DemoError("Athlete accounts are read-only.")
            username, training_id = parts[1], parts[2]
            _assert_access(username, current_user, role)
            return _update_training(username, training_id, data or {})
        if len(parts) == 3 and method == "DELETE":
            if role != "coach":
                raise DemoError("Athlete accounts are read-only.")
            username, training_id = parts[1], parts[2]
            _assert_access(username, current_user, role)
            return _delete_training(username, training_id)
        if len(parts) == 4 and method == "POST":
            if role != "coach":
                raise DemoError("Athlete accounts are read-only.")
            username, training_id, status = parts[1], parts[2], parts[3]
            _assert_access(username, current_user, role)
            return _set_training_status(username, training_id, status)

    if parts[0] == "profile":
        if len(parts) < 2:
            raise DemoError("Invalid profile endpoint.")
        username = parts[1]
        _assert_access(username, current_user, role, read_only=(role != "coach"))
        if len(parts) == 2 and method == "GET":
            return _get_profile(username)
        if len(parts) == 2 and method == "PUT":
            if role != "coach":
                raise DemoError("Only coaches can update profiles.")
            return _update_profile(username, data or {})
        raise DemoError("Profile import is disabled in the demo.")

    if parts[0] == "calendar":
        username = target_username or current_user
        _assert_access(username, current_user, role)
        if len(parts) == 3 and parts[1] == "day" and method == "GET":
            return _calendar_day(username, parts[2])
        if len(parts) == 3 and method == "GET":
            year = int(parts[1])
            month = int(parts[2])
            return _calendar_month(username, year, month)

    if parts[0] == "dashboard" and method == "GET":
        username = target_username or current_user
        _assert_access(username, current_user, role)
        return _dashboard(username)

    if parts[:3] == ["api", "coach", "settings"]:
        if role != "coach":
            raise DemoError("Access denied.")
        if method == "GET":
            return _get_coach_settings(current_user)
        if method == "POST":
            return _set_coach_settings(current_user, data or {})

    raise DemoError("Unknown endpoint.")


def _split_endpoint(endpoint: str) -> Tuple[str, Dict[str, str]]:
    raw = endpoint.strip()
    if raw.startswith("/"):
        raw = raw[1:]
    if "?" not in raw:
        return raw, {}
    path, query_string = raw.split("?", 1)
    query = {}
    for pair in query_string.split("&"):
        if not pair:
            continue
        key, _, val = pair.partition("=")
        query[key] = val
    return path, query


def _list_trainees(coach_username: str) -> List[str]:
    data = _load_json(ATHLETES_FILE, {"athletes": []})
    return [a["username"] for a in data.get("athletes", []) if a.get("coach_username") == coach_username]


def _assert_access(username: str, current_user: str, role: str, read_only: bool = False) -> None:
    if role == "coach":
        trainees = _list_trainees(current_user)
        if username not in trainees:
            raise DemoError("Access denied.")
        return
    if username != current_user:
        raise DemoError("Access denied.")
    if read_only:
        return


def _load_trainings() -> List[Dict[str, Any]]:
    return _load_json(TRAININGS_FILE, {"trainings": []}).get("trainings", [])


def _save_trainings(trainings: List[Dict[str, Any]]) -> None:
    _save_json(TRAININGS_FILE, {"trainings": trainings})


def _list_trainings(username: str) -> List[Dict[str, Any]]:
    with _LOCK:
        trainings = [t for t in _load_trainings() if t.get("username") == username]
        trainings.sort(key=lambda t: (t.get("date", ""), t.get("time", "")))
        return trainings


def _add_training(username: str, payload: Dict[str, Any]) -> Dict[str, Any]:
    _check_horizon(payload.get("date"))
    with _LOCK:
        trainings = _load_trainings()
        plan = _plan_from_description(payload.get("description", ""))
        entry = {
            "id": str(uuid.uuid4()),
            "username": username,
            "date": payload.get("date"),
            "time": payload.get("time", ""),
            "sport": payload.get("sport", "Swimming"),
            "description": payload.get("description", ""),
            "training_type": plan.get("training_type", "aerobic"),
            "training_plan": plan,
            "training_text": plan.get("training_text", ""),
            "status": "approved",
            "source": "manual",
        }
        trainings.append(entry)
        _save_trainings(trainings)
        return entry


def _update_training(username: str, training_id: str, payload: Dict[str, Any]) -> Dict[str, Any]:
    _check_horizon(payload.get("date"))
    with _LOCK:
        trainings = _load_trainings()
        for tr in trainings:
            if tr.get("id") == training_id and tr.get("username") == username:
                tr.update(
                    {
                        "date": payload.get("date", tr.get("date")),
                        "time": payload.get("time", tr.get("time")),
                        "sport": payload.get("sport", tr.get("sport")),
                        "description": payload.get("description", tr.get("description")),
                    }
                )
                plan = _plan_from_description(tr.get("description", ""))
                tr["training_type"] = plan.get("training_type", tr.get("training_type"))
                tr["training_plan"] = plan
                tr["training_text"] = plan.get("training_text", "")
                _save_trainings(trainings)
                return tr
    raise DemoError("Training not found.")


def _delete_training(username: str, training_id: str) -> Dict[str, str]:
    with _LOCK:
        trainings = _load_trainings()
        new_trainings = [t for t in trainings if not (t.get("id") == training_id and t.get("username") == username)]
        if len(new_trainings) == len(trainings):
            raise DemoError("Training not found.")
        _save_trainings(new_trainings)
        return {"status": "deleted"}


def _set_training_status(username: str, training_id: str, status: str) -> Dict[str, Any]:
    with _LOCK:
        trainings = _load_trainings()
        for tr in trainings:
            if tr.get("id") == training_id and tr.get("username") == username:
                if status == "reject":
                    trainings = [t for t in trainings if t.get("id") != training_id]
                    _save_trainings(trainings)
                    return {"status": "rejected"}
                tr["status"] = status
                _save_trainings(trainings)
                return tr
    raise DemoError("Training not found.")


def _generate_training(username: str, payload: Dict[str, Any]) -> Dict[str, Any]:
    date_value = payload.get("date")
    _check_horizon(date_value)
    athletes = _load_json(ATHLETES_FILE, {"athletes": []}).get("athletes", [])
    athlete = next((a for a in athletes if a.get("username") == username), None)
    if not athlete:
        raise DemoError("Athlete not found.")
    plan = build_training_plan(athlete, date_value, coach_brief=payload.get("coach_brief_text", ""))
    entry = {
        "id": str(uuid.uuid4()),
        "username": username,
        "date": date_value,
        "time": _default_time_for_date(date_value),
        "sport": "Swimming",
        "description": plan.get("summary", {}).get("main_goal") or plan.get("title"),
        "training_type": plan.get("training_type", "aerobic"),
        "training_plan": plan,
        "training_text": plan.get("training_text", ""),
        "status": "draft",
        "source": "generated",
    }
    with _LOCK:
        trainings = _load_trainings()
        trainings.append(entry)
        _save_trainings(trainings)
        return entry


def _get_profile(username: str) -> Dict[str, Any]:
    athletes = _load_json(ATHLETES_FILE, {"athletes": []}).get("athletes", [])
    athlete = next((a for a in athletes if a.get("username") == username), None)
    if not athlete:
        raise DemoError("Profile not found.")
    return {
        "discipline": athlete.get("discipline", ""),
        "personal_bests": athlete.get("personal_bests", {}),
    }


def _update_profile(username: str, payload: Dict[str, Any]) -> Dict[str, Any]:
    with _LOCK:
        athletes_data = _load_json(ATHLETES_FILE, {"athletes": []})
        updated = None
        for athlete in athletes_data.get("athletes", []):
            if athlete.get("username") == username:
                athlete["discipline"] = payload.get("discipline", athlete.get("discipline", ""))
                athlete["personal_bests"] = payload.get("personal_bests", athlete.get("personal_bests", {}))
                updated = athlete
                break
        if not updated:
            raise DemoError("Profile not found.")
        _save_json(ATHLETES_FILE, athletes_data)
        return updated


def _calendar_month(username: str, year: int, month: int) -> Dict[str, List[Dict[str, Any]]]:
    calendar_map: Dict[str, List[Dict[str, Any]]] = {}
    for tr in _list_trainings(username):
        date_str = tr.get("date")
        if not date_str:
            continue
        try:
            dt = datetime.strptime(date_str, "%Y-%m-%d").date()
        except Exception:
            continue
        if dt.year != year or dt.month != month:
            continue
        calendar_map.setdefault(date_str, []).append(
            {
                "name": tr.get("training_plan", {}).get("title") or tr.get("description") or "Training",
                "type": tr.get("training_type", "aerobic"),
            }
        )
    return calendar_map


def _calendar_day(username: str, iso_date: str) -> List[Dict[str, Any]]:
    items = []
    for tr in _list_trainings(username):
        if tr.get("date") == iso_date:
            items.append(
                {
                    "name": tr.get("training_plan", {}).get("title") or tr.get("description") or "Training",
                    "type": tr.get("training_type", "aerobic"),
                }
            )
    return items


def _dashboard(username: str) -> Dict[str, Any]:
    trainings = _list_trainings(username)
    today = date.today()
    week_start = today - timedelta(days=6)
    weekly_load = 0
    zone_totals = {"Z1": 0, "Z2": 0, "Z3": 0, "Z4": 0, "Z5": 0}
    for tr in trainings:
        date_str = tr.get("date")
        if not date_str:
            continue
        try:
            dt = datetime.strptime(date_str, "%Y-%m-%d").date()
        except Exception:
            continue
        if week_start <= dt <= today:
            plan = tr.get("training_plan", {}) or {}
            weekly_load += int(plan.get("total_distance_m") or 0)
            zones = plan.get("summary", {}).get("zones", {})
            for key in zone_totals:
                zone_totals[key] += int(zones.get(key) or 0)

    athletes = _load_json(ATHLETES_FILE, {"athletes": []}).get("athletes", [])
    athlete = next((a for a in athletes if a.get("username") == username), {})
    pb = _best_pb_for_discipline(athlete)
    now = date.today()
    monthly_sessions = sum(
        1
        for tr in trainings
        if tr.get("date", "").startswith(f"{now.year}-{now.month:02d}")
    )
    return {
        "weekly_load": f"{weekly_load} m",
        "last_pb": pb or "-",
        "monthly_sessions": monthly_sessions,
        "zone_distribution": zone_totals,
    }


def _get_coach_settings(username: str) -> Dict[str, Any]:
    users_data = _load_json(USERS_FILE, {"users": []})
    for user in users_data.get("users", []):
        if user.get("username") == username:
            return user.get("coach_settings", {})
    return {}


def _set_coach_settings(username: str, payload: Dict[str, Any]) -> Dict[str, Any]:
    with _LOCK:
        users_data = _load_json(USERS_FILE, {"users": []})
        for user in users_data.get("users", []):
            if user.get("username") == username:
                user["coach_settings"] = payload
                _save_json(USERS_FILE, users_data)
                return payload
    raise DemoError("Coach not found.")


def _check_horizon(date_value: Optional[str]) -> None:
    if not date_value:
        raise DemoError("Date is required.")
    try:
        dt = datetime.strptime(date_value, "%Y-%m-%d").date()
    except Exception:
        raise DemoError("Invalid date.")
    if dt > date.today() + timedelta(days=DEMO_HORIZON_DAYS):
        raise DemoError(f"Planning horizon is limited to {DEMO_HORIZON_DAYS} days in the demo.")


def _seed_users_athletes() -> Tuple[List[Dict[str, Any]], List[Dict[str, Any]]]:
    users = [
        {
            "username": "coach_demo",
            "password": "demo",
            "role": "coach",
            "display_name": "Coach Demo",
            "coach_settings": {},
        },
        {
            "username": "athlete_nina",
            "password": "demo",
            "role": "trainee",
            "display_name": "Nina Novak",
        },
        {
            "username": "athlete_peter",
            "password": "demo",
            "role": "trainee",
            "display_name": "Peter Hronec",
        },
        {
            "username": "athlete_lucia",
            "password": "demo",
            "role": "trainee",
            "display_name": "Lucia Varga",
        },
        {
            "username": "athlete_adam",
            "password": "demo",
            "role": "trainee",
            "display_name": "Adam Kovac",
        },
    ]

    athletes = [
        {
            "username": "athlete_nina",
            "first_name": "Nina",
            "last_name": "Novak",
            "gender": "F",
            "birth_year": 2009,
            "discipline": "200 Free",
            "performance_level": "developing",
            "training_volume": 18000,
            "fatigue": 0.25,
            "readiness": 0.8,
            "coach_username": "coach_demo",
            "personal_bests": {"50 Free": "00:30.20", "100 Free": "01:06.80", "200 Free": "02:22.50"},
        },
        {
            "username": "athlete_peter",
            "first_name": "Peter",
            "last_name": "Hronec",
            "gender": "M",
            "birth_year": 2007,
            "discipline": "100 Back",
            "performance_level": "regional",
            "training_volume": 23000,
            "fatigue": 0.45,
            "readiness": 0.65,
            "coach_username": "coach_demo",
            "personal_bests": {"50 Back": "00:28.90", "100 Back": "01:01.40"},
        },
        {
            "username": "athlete_lucia",
            "first_name": "Lucia",
            "last_name": "Varga",
            "gender": "F",
            "birth_year": 2005,
            "discipline": "50 Fly",
            "performance_level": "national",
            "training_volume": 26000,
            "fatigue": 0.6,
            "readiness": 0.5,
            "coach_username": "coach_demo",
            "personal_bests": {"50 Fly": "00:27.80", "100 Fly": "01:02.50"},
        },
        {
            "username": "athlete_adam",
            "first_name": "Adam",
            "last_name": "Kovac",
            "gender": "M",
            "birth_year": 2002,
            "discipline": "400 IM",
            "performance_level": "elite",
            "training_volume": 32000,
            "fatigue": 0.7,
            "readiness": 0.45,
            "coach_username": "coach_demo",
            "personal_bests": {"200 IM": "02:05.40", "400 IM": "04:32.10"},
        },
    ]
    return users, athletes


def _seed_trainings(athletes: List[Dict[str, Any]]) -> List[Dict[str, Any]]:
    sessions = []
    today = date.today()
    start = today - timedelta(days=14)
    end = today + timedelta(days=6)
    for athlete in athletes:
        current = start
        while current <= end:
            if current.weekday() in {0, 1, 3, 4, 5}:
                plan = build_training_plan(athlete, current.isoformat())
                sessions.append(
                    {
                        "id": str(uuid.uuid4()),
                        "username": athlete["username"],
                        "date": current.isoformat(),
                        "time": _default_time_for_weekday(current.weekday()),
                        "sport": "Swimming",
                        "description": plan.get("summary", {}).get("main_goal") or plan.get("title"),
                        "training_type": plan.get("training_type", "aerobic"),
                        "training_plan": plan,
                        "training_text": plan.get("training_text", ""),
                        "status": "approved" if current <= today else "planned",
                        "source": "seed",
                    }
                )
            current += timedelta(days=1)
    return sessions


def _default_time_for_date(date_value: str) -> str:
    try:
        dt = datetime.strptime(date_value, "%Y-%m-%d").date()
        return _default_time_for_weekday(dt.weekday())
    except Exception:
        return "16:00"


def _default_time_for_weekday(weekday: int) -> str:
    return {
        0: "16:00",
        1: "17:00",
        2: "16:30",
        3: "17:30",
        4: "16:15",
        5: "09:30",
        6: "08:30",
    }.get(weekday, "16:00")


def _best_pb_for_discipline(athlete: Dict[str, Any]) -> str:
    discipline = athlete.get("discipline")
    pbs = athlete.get("personal_bests", {})
    if discipline and discipline in pbs:
        return f"{discipline}: {pbs[discipline]}"
    if pbs:
        key = sorted(pbs.keys())[0]
        return f"{key}: {pbs[key]}"
    return ""


def _plan_from_description(description: str) -> Dict[str, Any]:
    total = _extract_volume(description)
    summary = {
        "main_goal": description.strip() or "Manual training entry",
        "total_distance_m": total,
        "zones": {"Z1": total, "Z2": 0, "Z3": 0, "Z4": 0, "Z5": 0},
    }
    lines = [description.strip()] if description.strip() else []
    training_text = _format_training_text("Manual Session", summary, lines)
    return {
        "title": "Manual Session",
        "training_type": "aerobic",
        "summary": summary,
        "lines": lines,
        "total_distance_m": total,
        "training_text": training_text,
    }


def build_training_plan(athlete: Dict[str, Any], date_value: str, coach_brief: str = "") -> Dict[str, Any]:
    dt = datetime.strptime(date_value, "%Y-%m-%d").date()
    day_type = _choose_day_type(dt, athlete)
    base_session = max(1200, int(athlete.get("training_volume", 18000) / 5))
    factor = {"aerobic": 1.0, "technique": 0.85, "main": 1.15, "speed": 0.95, "recovery": 0.6}.get(day_type, 1.0)
    fatigue = float(athlete.get("fatigue", 0.4))
    readiness = float(athlete.get("readiness", 0.6))
    adjust = max(0.8, min(1.05, 1.0 - fatigue * 0.1 + (readiness - 0.5) * 0.1))
    level = str(athlete.get("performance_level", "developing")).lower()
    level_factor = {"developing": 0.9, "regional": 1.0, "national": 1.05, "elite": 1.1}.get(level, 1.0)
    target = _round_distance(base_session * factor * adjust * level_factor)

    blocks = _plan_blocks(day_type, athlete)
    scaled_blocks = _scale_blocks(blocks, target)
    total = sum(b["distance"] for b in scaled_blocks)
    zones = {"Z1": 0, "Z2": 0, "Z3": 0, "Z4": 0, "Z5": 0}
    lines = []
    for block in scaled_blocks:
        zones[block["zone"]] += block["distance"]
        lines.append(f"{block['label']}: {block['text']} ({block['zone']})")

    summary = {
        "main_goal": _main_goal(day_type, athlete, coach_brief),
        "total_distance_m": total,
        "zones": zones,
    }
    title = f"{day_type.title()} session"
    training_text = _format_training_text(title, summary, lines)
    return {
        "title": title,
        "training_type": day_type,
        "summary": summary,
        "lines": lines,
        "total_distance_m": total,
        "training_text": training_text,
    }


def _choose_day_type(day: date, athlete: Dict[str, Any]) -> str:
    discipline = athlete.get("discipline", "").lower()
    sprint_focus = any(x in discipline for x in ["50", "100"])
    distance_focus = any(x in discipline for x in ["400", "800", "1500", "im"])
    weekday = day.weekday()
    if weekday == 0:
        return "aerobic"
    if weekday == 1:
        return "technique"
    if weekday == 2:
        return "main"
    if weekday == 3:
        return "aerobic" if distance_focus else "main"
    if weekday == 4:
        return "speed" if sprint_focus else "main"
    if weekday == 5:
        return "technique"
    return "recovery"


def _plan_blocks(day_type: str, athlete: Dict[str, Any]) -> List[Dict[str, Any]]:
    style = athlete.get("discipline", "Freestyle")
    if day_type == "technique":
        return [
            {"label": "WU", "distance": 400, "zone": "Z1", "text": "4x100 easy"},
            {"label": "Drill", "distance": 400, "zone": "Z1", "text": "8x50 drill + kick"},
            {"label": "Skills", "distance": 600, "zone": "Z2", "text": f"6x100 {style} focus"},
            {"label": "Main", "distance": 300, "zone": "Z3", "text": "6x50 build"},
            {"label": "CD", "distance": 200, "zone": "Z1", "text": "200 easy"},
        ]
    if day_type == "main":
        return [
            {"label": "WU", "distance": 400, "zone": "Z1", "text": "400 easy + 4x50 build"},
            {"label": "Prep", "distance": 300, "zone": "Z2", "text": "6x50 pace"},
            {"label": "Main", "distance": 1200, "zone": "Z4", "text": "8x100 race pace"},
            {"label": "Speed", "distance": 200, "zone": "Z5", "text": "4x50 sprint"},
            {"label": "CD", "distance": 200, "zone": "Z1", "text": "200 easy"},
        ]
    if day_type == "speed":
        return [
            {"label": "WU", "distance": 300, "zone": "Z1", "text": "300 easy"},
            {"label": "Drill", "distance": 300, "zone": "Z2", "text": "6x50 kick"},
            {"label": "Main", "distance": 600, "zone": "Z4", "text": "12x25 power"},
            {"label": "Speed", "distance": 300, "zone": "Z5", "text": "6x50 sprint"},
            {"label": "CD", "distance": 200, "zone": "Z1", "text": "200 easy"},
        ]
    if day_type == "recovery":
        return [
            {"label": "WU", "distance": 300, "zone": "Z1", "text": "300 easy"},
            {"label": "Flow", "distance": 400, "zone": "Z1", "text": "8x50 relaxed"},
            {"label": "Skills", "distance": 400, "zone": "Z2", "text": "4x100 smooth"},
            {"label": "CD", "distance": 200, "zone": "Z1", "text": "200 easy"},
        ]
    return [
        {"label": "WU", "distance": 400, "zone": "Z1", "text": "400 easy"},
        {"label": "Drill", "distance": 400, "zone": "Z2", "text": "8x50 drill"},
        {"label": "Main", "distance": 1200, "zone": "Z2", "text": "3x400 aerobic"},
        {"label": "Build", "distance": 400, "zone": "Z3", "text": "4x100 build"},
        {"label": "CD", "distance": 200, "zone": "Z1", "text": "200 easy"},
    ]


def _scale_blocks(blocks: List[Dict[str, Any]], target: int) -> List[Dict[str, Any]]:
    base = sum(b["distance"] for b in blocks) or 1
    scale = target / base
    scaled = []
    for block in blocks:
        dist = _round_distance(block["distance"] * scale)
        scaled.append({**block, "distance": dist})
    diff = target - sum(b["distance"] for b in scaled)
    if scaled:
        scaled[-1]["distance"] = max(50, scaled[-1]["distance"] + diff)
        scaled[-1]["text"] = _adjust_text_distance(scaled[-1]["text"], scaled[-1]["distance"])
    return scaled


def _adjust_text_distance(text: str, distance: int) -> str:
    if "x" in text:
        return text
    return f"{distance}m easy"


def _main_goal(day_type: str, athlete: Dict[str, Any], coach_brief: str) -> str:
    discipline = athlete.get("discipline", "Freestyle")
    focus = {
        "technique": f"Technique focus for {discipline}",
        "aerobic": f"Aerobic base for {discipline}",
        "main": f"Race pace main set for {discipline}",
        "speed": f"Speed & power for {discipline}",
        "recovery": f"Recovery and feel for {discipline}",
    }.get(day_type, f"Balanced session for {discipline}")
    if coach_brief:
        return f"{focus}. Coach note: {coach_brief}"
    return focus


def _format_training_text(title: str, summary: Dict[str, Any], lines: List[str]) -> str:
    zones = summary.get("zones", {})
    zone_line = " | ".join([f"{z}: {zones.get(z, 0)}m" for z in ["Z1", "Z2", "Z3", "Z4", "Z5"]])
    parts = [
        title,
        f"Goal: {summary.get('main_goal', '')}",
        f"Total: {summary.get('total_distance_m', 0)} m",
        f"Zones: {zone_line}",
        "",
    ]
    parts.extend(lines)
    return "\n".join([p for p in parts if p is not None])


def _extract_volume(description: str) -> int:
    if not description:
        return 0
    total = 0
    for count, dist in re.findall(r"(\\d+)\\s*[xX]\\s*(\\d+)", description):
        total += int(count) * int(dist)
    for meters in re.findall(r"(\\d{2,4})\\s*[Rr]?", description):
        total += int(meters)
    return total


def _round_distance(value: float) -> int:
    return int(round(value / 50.0) * 50)
